/*
 * Copyright 2017 - 2025 the original author or authors.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see [https://www.gnu.org/licenses/]
 */

package infra.context.properties.source;

import org.jspecify.annotations.Nullable;

import java.util.Locale;
import java.util.Map;
import java.util.Random;

import infra.context.properties.source.ConfigurationPropertyName.Form;
import infra.core.env.EnumerablePropertySource;
import infra.core.env.PropertySource;
import infra.core.env.StandardEnvironment;
import infra.core.env.SystemEnvironmentPropertySource;
import infra.lang.Assert;
import infra.origin.Origin;
import infra.origin.PropertySourceOrigin;

/**
 * {@link ConfigurationPropertySource} backed by a non-enumerable Framework
 * {@link PropertySource} or a restricted {@link EnumerablePropertySource} implementation
 * (such as a security restricted {@code systemEnvironment} source). A
 * {@link PropertySource} is adapted with the help of a {@link PropertyMapper} which
 * provides the mapping rules for individual properties.
 * <p>
 * Each {@link ConfigurationPropertySource#getConfigurationProperty
 * getConfigurationProperty} call attempts to
 * {@link PropertyMapper#map(ConfigurationPropertyName) map} the
 * {@link ConfigurationPropertyName} to one or more {@code String} based names. This
 * allows fast property resolution for well formed property sources.
 * <p>
 * When possible the {@link DefaultIterableConfigurationPropertySource} will be used in
 * preference to this implementation since it supports full "relaxed" style resolution.
 *
 * @author Phillip Webb
 * @author Madhura Bhave
 * @author <a href="https://github.com/TAKETODAY">Harry Yang</a>
 * @see #from(PropertySource)
 * @see PropertyMapper
 * @see DefaultIterableConfigurationPropertySource
 * @since 4.0
 */
@SuppressWarnings("NullAway")
class DefaultConfigurationPropertySource implements ConfigurationPropertySource {

  private static final PropertyMapper[] DEFAULT_MAPPERS = {
          DefaultPropertyMapper.INSTANCE
  };

  private static final PropertyMapper[] SYSTEM_ENVIRONMENT_MAPPERS = {
          SystemEnvironmentPropertyMapper.INSTANCE,
          DefaultPropertyMapper.INSTANCE
  };

  private final PropertySource<?> propertySource;

  private final boolean systemEnvironmentSource;

  private final PropertyMapper[] mappers;

  /**
   * Create a new {@link DefaultConfigurationPropertySource} implementation.
   *
   * @param propertySource the source property source
   * @param mappers the property mappers
   */
  DefaultConfigurationPropertySource(PropertySource<?> propertySource, boolean systemEnvironmentSource, PropertyMapper... mappers) {
    Assert.notNull(propertySource, "PropertySource is required");
    Assert.isTrue(mappers.length > 0, "Mappers must contain at least one item");
    this.systemEnvironmentSource = systemEnvironmentSource;
    this.propertySource = propertySource;
    this.mappers = mappers;
  }

  @Nullable
  @Override
  public ConfigurationProperty getConfigurationProperty(@Nullable ConfigurationPropertyName name) {
    if (name == null) {
      return null;
    }
    for (PropertyMapper mapper : this.mappers) {
      try {
        for (String candidate : mapper.map(name)) {
          Object value = getPropertySourceProperty(candidate);
          if (value != null) {
            Origin origin = PropertySourceOrigin.get(this.propertySource, candidate);
            return ConfigurationProperty.of(this, name, value, origin);
          }
        }
      }
      catch (Exception ignored) {
      }
    }
    return null;
  }

  @Nullable
  protected final Object getPropertySourceProperty(String name) {
    // Save calls to SystemEnvironmentPropertySource.resolvePropertyName(...)
    // since we've already done the mapping
    PropertySource<?> propertySource = getPropertySource();
    return (!this.systemEnvironmentSource) ? propertySource.getProperty(name)
            : getSystemEnvironmentProperty(((SystemEnvironmentPropertySource) propertySource).getSource(), name);
  }

  Object getSystemEnvironmentProperty(Map<String, Object> systemEnvironment, String name) {
    Object value = systemEnvironment.get(name);
    return value != null ? value : systemEnvironment.get(name.toLowerCase(Locale.ROOT));
  }

  @Override
  public ConfigurationPropertyState containsDescendantOf(ConfigurationPropertyName name) {
    PropertySource<?> source = getPropertySource();
    if (source.getSource() instanceof Random) {
      return containsDescendantOfForRandom("random", name);
    }
    if (source.getSource() instanceof PropertySource<?>
            && ((PropertySource<?>) source.getSource()).getSource() instanceof Random) {
      // Assume wrapped random sources use the source name as the prefix
      return containsDescendantOfForRandom(source.getName(), name);
    }
    return ConfigurationPropertyState.UNKNOWN;
  }

  private static ConfigurationPropertyState containsDescendantOfForRandom(
          String prefix, ConfigurationPropertyName name) {
    if (name.getNumberOfElements() > 1 && name.getElement(0, Form.DASHED).equals(prefix)) {
      return ConfigurationPropertyState.PRESENT;
    }
    return ConfigurationPropertyState.ABSENT;
  }

  @Override
  public Object getUnderlyingSource() {
    return this.propertySource;
  }

  protected PropertySource<?> getPropertySource() {
    return this.propertySource;
  }

  protected final boolean isSystemEnvironmentSource() {
    return this.systemEnvironmentSource;
  }

  protected final PropertyMapper[] getMappers() {
    return this.mappers;
  }

  @Override
  public String toString() {
    return this.propertySource.toString();
  }

  /**
   * Create a new {@link DefaultConfigurationPropertySource} for the specified
   * {@link PropertySource}.
   *
   * @param source the source Framework {@link PropertySource}
   * @return a {@link DefaultConfigurationPropertySource} or
   * {@link DefaultIterableConfigurationPropertySource} instance
   */
  static DefaultConfigurationPropertySource from(PropertySource<?> source) {
    Assert.notNull(source, "Source is required");
    boolean systemEnvironmentSource = isSystemEnvironmentPropertySource(source);
    PropertyMapper[] mappers = (!systemEnvironmentSource) ? DEFAULT_MAPPERS : SYSTEM_ENVIRONMENT_MAPPERS;
    return (!isFullEnumerable(source))
            ? new DefaultConfigurationPropertySource(source, systemEnvironmentSource, mappers)
            : new DefaultIterableConfigurationPropertySource((EnumerablePropertySource<?>) source,
                    systemEnvironmentSource, mappers);
  }

  private static boolean isSystemEnvironmentPropertySource(PropertySource<?> source) {
    String name = source.getName();
    return (source instanceof SystemEnvironmentPropertySource)
            && (StandardEnvironment.SYSTEM_ENVIRONMENT_PROPERTY_SOURCE_NAME.equals(name)
            || name.endsWith("-" + StandardEnvironment.SYSTEM_ENVIRONMENT_PROPERTY_SOURCE_NAME));
  }

  private static boolean isFullEnumerable(PropertySource<?> source) {
    PropertySource<?> rootSource = getRootSource(source);
    if (rootSource.getSource() instanceof Map map) {
      // Check we're not security restricted
      try {
        map.size();
      }
      catch (UnsupportedOperationException ex) {
        return false;
      }
    }
    return (source instanceof EnumerablePropertySource);
  }

  private static PropertySource<?> getRootSource(PropertySource<?> source) {
    while (source.getSource() instanceof PropertySource) {
      source = (PropertySource<?>) source.getSource();
    }
    return source;
  }

}
