/*
 * Copyright 2017 - 2025 the original author or authors.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see [https://www.gnu.org/licenses/]
 */

package infra.context.condition;

import org.jspecify.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import infra.beans.BeansException;
import infra.beans.factory.BeanClassLoaderAware;
import infra.beans.factory.BeanFactory;
import infra.beans.factory.BeanFactoryAware;
import infra.context.annotation.config.AutoConfigurationImportFilter;
import infra.context.annotation.config.AutoConfigurationMetadata;
import infra.util.ClassUtils;
import infra.util.CollectionUtils;

/**
 * Abstract base class for a {@link InfraCondition} that also implements
 * {@link AutoConfigurationImportFilter}.
 *
 * @author Phillip Webb
 * @author <a href="https://github.com/TAKETODAY">Harry Yang</a>
 * @since 4.0 2022/1/16 16:07
 */
public abstract class FilteringInfraCondition extends InfraCondition
        implements AutoConfigurationImportFilter, BeanFactoryAware, BeanClassLoaderAware {

  @SuppressWarnings("NullAway.Init")
  private BeanFactory beanFactory;

  @SuppressWarnings("NullAway.Init")
  private ClassLoader beanClassLoader;

  @Override
  public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
    this.beanFactory = beanFactory;
  }

  protected final BeanFactory getBeanFactory() {
    return this.beanFactory;
  }

  protected final ClassLoader getBeanClassLoader() {
    return this.beanClassLoader;
  }

  @Override
  public void setBeanClassLoader(ClassLoader classLoader) {
    this.beanClassLoader = classLoader;
  }

  @Override
  public boolean[] match(String[] autoConfigurationClasses, AutoConfigurationMetadata configMetadata) {
    ConditionEvaluationReport report = ConditionEvaluationReport.find(this.beanFactory);
    @Nullable ConditionOutcome[] outcomes = getOutcomes(autoConfigurationClasses, configMetadata);
    boolean[] match = new boolean[outcomes.length];
    for (int i = 0; i < outcomes.length; i++) {
      ConditionOutcome outcome = outcomes[i];
      match[i] = (outcome == null || outcome.isMatch());
      if (!match[i] && outcome != null) {
        String autoConfigurationClass = autoConfigurationClasses[i];
        logOutcome(autoConfigurationClass, outcome);
        if (report != null) {
          report.recordConditionEvaluation(autoConfigurationClass, this, outcome);
        }
      }
    }
    return match;
  }

  protected abstract @Nullable ConditionOutcome[] getOutcomes(String[] configClasses, AutoConfigurationMetadata configMetadata);

  protected final List<String> filter(@Nullable Collection<String> classNames, ClassNameFilter classNameFilter, @Nullable ClassLoader classLoader) {
    if (CollectionUtils.isEmpty(classNames)) {
      return Collections.emptyList();
    }
    ArrayList<String> matches = new ArrayList<>(classNames.size());
    for (String candidate : classNames) {
      if (classNameFilter.matches(candidate, classLoader)) {
        matches.add(candidate);
      }
    }
    return matches;
  }

  /**
   * Slightly faster variant of {@link ClassUtils#forName(String, ClassLoader)} that
   * doesn't deal with primitives, arrays or inner types.
   *
   * @param className the class name to resolve
   * @param classLoader the class loader to use
   * @return a resolved class
   * @throws ClassNotFoundException if the class cannot be found
   */
  protected static Class<?> resolve(String className, @Nullable ClassLoader classLoader) throws ClassNotFoundException {
    if (classLoader != null) {
      return Class.forName(className, false, classLoader);
    }
    return Class.forName(className);
  }

  protected enum ClassNameFilter {

    PRESENT {
      @Override
      public boolean matches(String className, @Nullable ClassLoader classLoader) {
        return isPresent(className, classLoader);
      }

    },

    MISSING {
      @Override
      public boolean matches(String className, @Nullable ClassLoader classLoader) {
        return !isPresent(className, classLoader);
      }

    };

    public abstract boolean matches(String className, @Nullable ClassLoader classLoader);

    private static boolean isPresent(String className, @Nullable ClassLoader classLoader) {
      if (classLoader == null) {
        classLoader = ClassUtils.getDefaultClassLoader();
      }
      try {
        resolve(className, classLoader);
        return true;
      }
      catch (Throwable ex) {
        return false;
      }
    }

  }
}
