/*
 * Copyright 2017 - 2025 the original author or authors.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see [https://www.gnu.org/licenses/]
 */

package infra.cache.interceptor;

import org.jspecify.annotations.Nullable;

/**
 * Abstract the invocation of a cache operation.
 *
 * <p>Does not provide a way to transmit checked exceptions but
 * provide a special exception that should be used to wrap any
 * exception that was thrown by the underlying invocation.
 * Callers are expected to handle this issue type specifically.
 *
 * @author Stephane Nicoll
 * @author <a href="https://github.com/TAKETODAY">Harry Yang</a>
 * @since 4.0
 */
@FunctionalInterface
public interface CacheOperationInvoker {

  /**
   * Invoke the cache operation defined by this instance. Wraps any exception
   * that is thrown during the invocation in a {@link ThrowableWrapper}.
   *
   * @return the result of the operation
   * @throws ThrowableWrapper if an error occurred while invoking the operation
   */
  @Nullable
  Object invoke() throws ThrowableWrapper;

  /**
   * Wrap any exception thrown while invoking {@link #invoke()}.
   */
  @SuppressWarnings("serial")
  class ThrowableWrapper extends RuntimeException {

    private final Throwable original;

    public ThrowableWrapper(Throwable original) {
      super(original.getMessage(), original);
      this.original = original;
    }

    public Throwable getOriginal() {
      return this.original;
    }
  }

}
