/*
 * Original Author -> Harry Yang (taketoday@foxmail.com) https://taketoday.cn
 * Copyright © TODAY & 2017 - 2021 All Rights Reserved.
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see [http://www.gnu.org/licenses/]
 */
package cn.taketoday.context;

/**
 * An extension of the {@link Lifecycle} interface for those objects that require
 * to be started upon {@code ApplicationContext} refresh and/or shutdown in a
 * particular order.
 *
 * <p>The {@link #isAutoStartup()} return value indicates whether this object should
 * be started at the time of a context refresh. The callback-accepting
 * {@link #stop(Runnable)} method is useful for objects that have an asynchronous
 * shutdown process. Any implementation of this interface <i>must</i> invoke the
 * callback's {@code run()} method upon shutdown completion to avoid unnecessary
 * delays in the overall {@code ApplicationContext} shutdown.
 *
 * <p>This interface extends {@link Phased}, and the {@link #getPhase()} method's
 * return value indicates the phase within which this {@code Lifecycle} component
 * should be started and stopped. The startup process begins with the <i>lowest</i>
 * phase value and ends with the <i>highest</i> phase value ({@code Integer.MIN_VALUE}
 * is the lowest possible, and {@code Integer.MAX_VALUE} is the highest possible).
 * The shutdown process will apply the reverse order. Any components with the
 * same value will be arbitrarily ordered within the same phase.
 *
 * <p>Example: if component B depends on component A having already started,
 * then component A should have a lower phase value than component B. During
 * the shutdown process, component B would be stopped before component A.
 *
 * <p>Any explicit "depends-on" relationship will take precedence over the phase
 * order such that the dependent bean always starts after its dependency and
 * always stops before its dependency.
 *
 * <p>Any {@code Lifecycle} components within the context that do not also
 * implement {@code SmartLifecycle} will be treated as if they have a phase
 * value of {@code 0}. This allows a {@code SmartLifecycle} component to start
 * before those {@code Lifecycle} components if the {@code SmartLifecycle}
 * component has a negative phase value, or the {@code SmartLifecycle} component
 * may start after those {@code Lifecycle} components if the {@code SmartLifecycle}
 * component has a positive phase value.
 *
 * <p>Note that, due to the auto-startup support in {@code SmartLifecycle}, a
 * {@code SmartLifecycle} bean instance will usually get initialized on startup
 * of the application context in any case. As a consequence, the bean definition
 * lazy-init flag has very limited actual effect on {@code SmartLifecycle} beans.
 *
 * @author Mark Fisher
 * @author Juergen Hoeller
 * @author Sam Brannen
 * @author Harry Yang 2021/11/12 16:28
 * @see LifecycleProcessor
 * @see ConfigurableApplicationContext
 * @since 4.0
 */
public interface SmartLifecycle extends Lifecycle, Phased {

  /**
   * The default phase for {@code SmartLifecycle}: {@code Integer.MAX_VALUE}.
   * <p>This is different from the common phase {@code 0} associated with regular
   * {@link Lifecycle} implementations, putting the typically auto-started
   * {@code SmartLifecycle} beans into a later startup phase and an earlier
   * shutdown phase.
   *
   * @see #getPhase()
   * @see cn.taketoday.context.support.DefaultLifecycleProcessor#getPhase(Lifecycle)
   */
  int DEFAULT_PHASE = Integer.MAX_VALUE;

  /**
   * Returns {@code true} if this {@code Lifecycle} component should get
   * started automatically by the container at the time that the containing
   * {@link ApplicationContext} gets refreshed.
   * <p>A value of {@code false} indicates that the component is intended to
   * be started through an explicit {@link #start()} call instead, analogous
   * to a plain {@link Lifecycle} implementation.
   * <p>The default implementation returns {@code true}.
   *
   * @see #start()
   * @see #getPhase()
   * @see LifecycleProcessor#onRefresh()
   * @see ConfigurableApplicationContext#refresh()
   */
  default boolean isAutoStartup() {
    return true;
  }

  /**
   * Indicates that a Lifecycle component must stop if it is currently running.
   * <p>The provided callback is used by the {@link LifecycleProcessor} to support
   * an ordered, and potentially concurrent, shutdown of all components having a
   * common shutdown order value. The callback <b>must</b> be executed after
   * the {@code SmartLifecycle} component does indeed stop.
   * <p>The {@link LifecycleProcessor} will call <i>only</i> this variant of the
   * {@code stop} method; i.e. {@link Lifecycle#stop()} will not be called for
   * {@code SmartLifecycle} implementations unless explicitly delegated to within
   * the implementation of this method.
   * <p>The default implementation delegates to {@link #stop()} and immediately
   * triggers the given callback in the calling thread. Note that there is no
   * synchronization between the two, so custom implementations may at least
   * want to put the same steps within their common lifecycle monitor (if any).
   *
   * @see #stop()
   * @see #getPhase()
   */
  default void stop(Runnable callback) {
    stop();
    callback.run();
  }

  /**
   * Return the phase that this lifecycle object is supposed to run in.
   * <p>The default implementation returns {@link #DEFAULT_PHASE} in order to
   * let {@code stop()} callbacks execute after regular {@code Lifecycle}
   * implementations.
   *
   * @see #isAutoStartup()
   * @see #start()
   * @see #stop(Runnable)
   * @see cn.taketoday.context.support.DefaultLifecycleProcessor#getPhase(Lifecycle)
   */
  @Override
  default int getPhase() {
    return DEFAULT_PHASE;
  }

}
