package cn.t.util.media.code;

import cn.t.util.media.code.qrcode.decoder.DefaultQRCodeImage;
import cn.t.util.media.code.qrcode.decoder.QRCodeDecoder;
import cn.t.util.media.code.qrcode.encoder.QrCodeTwoDimensionCodeConfig;
import cn.t.util.media.code.qrcode.encoder.Qrcode;
import cn.t.util.media.code.zxing.BufferedImageLuminanceSource;
import cn.t.util.media.code.zxing.MatrixToImageWriter;
import cn.t.util.media.code.zxing.ZxingTwoDimensionCodeConfig;
import com.google.zxing.*;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.common.HybridBinarizer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;

public class TwoDimensionCodeUtil {

    private static final Logger logger = LoggerFactory.getLogger(TwoDimensionCodeUtil.class);

    /**
     * zxing输出二维码
     * @param codeConfig xxx
     * @param content xxx
     * @param path xxx
     */
    public static void zxingWriteCode(ZxingTwoDimensionCodeConfig codeConfig, String content, String path) {
        //定义二维码的参数
        HashMap<EncodeHintType, Object> hints = new HashMap<>();
        hints.put(EncodeHintType.CHARACTER_SET, codeConfig.getCharset());
        //就错级别(中等)
        hints.put(EncodeHintType.ERROR_CORRECTION, codeConfig.getErrorCorrectionLevel());
        //边距(默认是5)
        hints.put(EncodeHintType.MARGIN, codeConfig.getMargin());
        try {
            BitMatrix bitMatrix = new MultiFormatWriter().encode(content, BarcodeFormat.QR_CODE, codeConfig.getWidth(), codeConfig.getHeight(), hints);
            Path file = new File(path).toPath();
            MatrixToImageWriter.writeToPath(bitMatrix, codeConfig.getFormat(), file);
        } catch (Exception e) {
            logger.error("二维码输出异常", e);
        }
    }

    /**
     * zxing读取二维码
     * @param codeConfig xxx
     * @param path xxx
     * @return xxx
     */
    public static Result zxingReadCode(ZxingTwoDimensionCodeConfig codeConfig, String path) {
        try {
            MultiFormatReader formatReader = new MultiFormatReader();
            File file = new File(path);
            BufferedImage image = ImageIO.read(file);
            BinaryBitmap binaryBitmap = new BinaryBitmap(new HybridBinarizer(new BufferedImageLuminanceSource(image)));
            //定义二维码的参数
            HashMap<DecodeHintType, Object> hints = new HashMap<>();
            hints.put(DecodeHintType.CHARACTER_SET, codeConfig.getCharset());
            Result result = formatReader.decode(binaryBitmap, hints);
            if (logger.isDebugEnabled()) {
                logger.debug("二维码格式类型: {}", result.getBarcodeFormat());
                logger.debug("二维码文本内容: {}", result.getText());
            }
            return result;
        } catch (IOException | NotFoundException e) {
            logger.error(e.toString());
        }
        return null;
    }

    /**
     * qr code输出二维码
     * @param codeConfig xxx
     * @param content xxx
     * @param path xxx
     * @throws IOException xxx
     */
    public static void qrCodeWriteCode(QrCodeTwoDimensionCodeConfig codeConfig, String content, String path) throws IOException {
        Qrcode x = new Qrcode();
        //纠错等级
        x.setQrcodeErrorCorrect(codeConfig.getErrorCorrectLevel());
        //N代表数字, A代表a-z, B代表其他字符
        x.setQrcodeEncodeMode(codeConfig.getQrcodeEncodeMode());
        //版本1-40
        x.setQrcodeVersion(codeConfig.getVersion());
        BufferedImage bufferedImage = new BufferedImage(codeConfig.getWidth(), codeConfig.getHeight(), BufferedImage.TYPE_INT_RGB);
        Graphics2D gs = bufferedImage.createGraphics();
        gs.setBackground(codeConfig.getBackgroundColor());
        gs.setColor(codeConfig.getFillColor());
        gs.clearRect(0, 0, codeConfig.getWidth(), codeConfig.getHeight());
        byte[] d = content.getBytes();
        if (d.length > 0 && d.length < 120) {
            boolean[][] s = x.calQrcode(d);
            for (int i = 0; i < s.length; i++) {
                for (int j = 0; j < s.length; j++) {
                    if (s[j][i]) {
                        gs.fillRect(j * 3 + codeConfig.getPixOff(), i * 3 + codeConfig.getPixOff(), 3, 3);
                    }
                }
            }
        }
        gs.dispose();
        bufferedImage.flush();
        ImageIO.write(bufferedImage, codeConfig.getFormat(), new File(path));
    }

    /**
     * qr code读取二维码
     * @param codeConfig xxx
     * @param path xxx
     * @return xxx
     */
    public static String qrCodeReadCode(QrCodeTwoDimensionCodeConfig codeConfig, String path) {
        try {
            File file = new File(path);
            BufferedImage bufferedImage = ImageIO.read(file);
            QRCodeDecoder codeDecoder = new QRCodeDecoder();
            byte[] decodeBytes = codeDecoder.decode(new DefaultQRCodeImage(bufferedImage));
            return new String(decodeBytes, codeConfig.getCharset());
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }
}


