package cn.t.util.io;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;

/**
 *
 */
public class FileUtil {

    private static final Logger logger = LoggerFactory.getLogger(FileUtil.class);

    /**
     * path 不以'/'开头时默认是从此类所在的包下取资源，以'/'开头则是从ClassPath(Src根目录)根下获取。
     * 其只是通过path构造一个绝对路径，最终还是由ClassLoader获取资源。
     * @param clazz xxx
     * @param path xxx
     * @return xxx
     */
    public static InputStream getResourceInputStream(Class clazz, String path) {
        logger.debug("load input stream by class {}, path: {}", clazz.getName(), path);
        return clazz.getResourceAsStream(path);
    }

    /**
     * 默认则是从ClassPath根下获取，path不能以'/'开头，最终是由ClassLoader获取资源。
     * @param classLoader xxx
     * @param path xxx
     * @return xxx
     */
    public static InputStream getResourceInputStream(ClassLoader classLoader, String path) {
        logger.debug("load input stream by classloader {}, path: {}", classLoader, path);
        return classLoader.getResourceAsStream(path);
    }

    public static FileInputStream getResourceFileInputStream(ClassLoader classLoader, String path) throws URISyntaxException, FileNotFoundException {
        logger.debug("load input stream by classloader {}, path: {}", classLoader, path);
        URL url = classLoader.getResource(path);
        if (url != null) {
            return new FileInputStream(new File(url.toURI()));
        }
        return null;
    }

    public static FileInputStream getFileInputStream(String path) throws FileNotFoundException {
        logger.debug("load input stream by file path {}", path);
        return new FileInputStream(path);
    }

    public static DataInputStream getResourceDataInputStream(String path) throws FileNotFoundException {
        logger.debug("load input stream by file path {}", path);
        return new DataInputStream(getFileInputStream(path));
    }

    public static FileReader getFileReader(String path) throws FileNotFoundException {
        if (logger.isDebugEnabled()) {
            logger.debug("load input stream by file path {}", path);
        }
        return new FileReader(path);
    }

    public static BufferedInputStream getBufferedInputStream(String path) throws FileNotFoundException {
        return new BufferedInputStream(getFileInputStream(path));
    }

    public static BufferedReader getBufferedReader(String path) throws FileNotFoundException {
        return new BufferedReader(getFileReader(path));
    }

    public static byte[] getFileBytes(String path) throws IOException {
        try (BufferedInputStream bis = getBufferedInputStream(path)) {
            byte[] content = new byte[bis.available()];
            int length = bis.read(content);
            if (logger.isDebugEnabled()) {
                logger.debug("read file length: {}", length);
            }
            return content;
        }
    }

    public static String getProjectClassPath() {
        return FileUtil.class.getResource("/").getPath().substring(1);
    }

    public static String getProjectPath() {
        return System.getProperty("user.dir");
    }

    public static String saveToTempDir(byte[] bytes, String contentType) throws IOException {
        String path = System.getProperty("java.io.tmpdir").concat(String.valueOf(System.currentTimeMillis())).concat(".").concat(analyseImageTail(contentType));
        File file = new File(path);
        boolean success = file.createNewFile();
        if (!success) {
            logger.error("cannot create new file at: {}", System.getProperty("java.io.tmpdir"));
            return null;
        }
        try (
            FileOutputStream fos = new FileOutputStream(file)
        ) {
            fos.write(bytes);
        }
        return path;
    }

    private static String analyseImageTail(String contentType) {
        String[] eles = contentType.split("/");
        return eles.length > 1 ? eles[1] : eles[0];
    }


    /**
     * 文件 zero copy 复制
     * @param from xxx
     * @param to xxx
     * @throws IOException xxx
     */
    public static void copyFile(String from, String to) throws IOException {
        try (
            FileInputStream fis = new FileInputStream(from);
            FileOutputStream fos = new FileOutputStream(to);
            FileChannel fci = fis.getChannel();
            FileChannel fco = fos.getChannel()
        ) {
            fci.transferTo(0, fci.size(), fco);
        }
    }

    /**
     * 获取文件mime type
     * @param path xxx
     * @return xxx
     * @throws IOException xxx
     */
    public static String getMimeType(String path) throws IOException {

        try (
            BufferedInputStream bis = getBufferedInputStream(path)
        ) {
            String mineType = URLConnection.guessContentTypeFromStream(bis);
            if (mineType == null) {
                return "application/octet-stream";
            } else {
                return mineType;
            }
        }
    }

    /**
     * 创建临时文件
     * @param prefix xxx
     * @param suffix xxx
     * @return xxx
     * @throws IOException xxx
     */
    public static Path createTempFile(String prefix, String suffix) throws IOException {
        return Files.createTempFile(prefix, suffix);
    }

    /**
     * 将输入流写入文件
     * @param in     xxx
     * @param prefix xxx
     * @param suffix xxx
     * @return xxx
     * @throws IOException xxx
     */
    public static Path copyInputStreamToTempFile(InputStream in, String prefix, String suffix) throws IOException {
        Path tmp = createTempFile(prefix, suffix);
        Files.copy(in, tmp, StandardCopyOption.REPLACE_EXISTING);
        return tmp;
    }

    public static String appendFilePath(String original, String append) {
        if(original.endsWith(File.separator)) {
            return original + append;
        } else {
            return original + File.separator + append;
        }
    }

    public static boolean initFile(File file) throws IOException {
        if(!file.exists()) {
            if(!initDirectory(file.getParentFile())) {
                return false;
            }
            return file.createNewFile();
        }
        return true;
    }

    public static boolean initDirectory(File file) throws IOException {
        if(!file.exists()) {
            return file.mkdirs();
        }
        return true;
    }

}
