package cn.t.util.doc;

import cn.t.util.common.StringUtil;
import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.ss.usermodel.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public class ExcelUtil {

    private static final Logger logger = LoggerFactory.getLogger(ExcelUtil.class);

    /**
     * 读取excel文档
     * @param path              xxx
     * @param readExcelCallBack xxx
     * @throws IOException xxx
     * @throws InvalidFormatException xxx
     */
    public static void readWorkbook(String path, ReadExcelCallBack readExcelCallBack) throws IOException, InvalidFormatException {
        readWorkbook(new File(path), readExcelCallBack);
    }

    /**
     * 读取excel文档
     * @param file              xxx
     * @param readExcelCallBack xxx
     * @throws IOException xxx
     * @throws InvalidFormatException xxx
     */
    private static void readWorkbook(File file, ReadExcelCallBack readExcelCallBack) throws IOException, InvalidFormatException {
        readSheet(WorkbookFactory.create(new FileInputStream(file)), readExcelCallBack);
    }

    /**
     * 读取sheet
     * @param workbook          xxx
     * @param readExcelCallBack xxx
     * @throws IOException xxx
     */
    private static void readSheet(Workbook workbook, ReadExcelCallBack readExcelCallBack) throws IOException {
        int totalSheet = workbook.getNumberOfSheets();
        for (int i = 0; i < totalSheet; i++) {
            Sheet currentSheet = workbook.getSheetAt(i);
            //读取当前sheet
            readExcelCallBack.readSheet(currentSheet);
            readRow(currentSheet, readExcelCallBack);
            //读取sheet完成
            readExcelCallBack.readSheetComplete(currentSheet);
        }
        readExcelCallBack.readWorkbookComplete(workbook);
    }

    /**
     * 读取row
     * @param sheet             xxx
     * @param readExcelCallBack xxx
     */
    private static void readRow(Sheet sheet, ReadExcelCallBack readExcelCallBack) {
        int totalRow = sheet.getLastRowNum() + 1;
        for (int i = 0; i < totalRow; i++) {
            Row currentRow = sheet.getRow(i);
            readExcelCallBack.readRow(currentRow, i);
            if (currentRow != null) {
                //读取当前row
                readCell(currentRow, readExcelCallBack);
                //读取Row完成
                readExcelCallBack.readRowComplete(currentRow);
            }
        }
    }

    /**
     * 读取cell
     * @param row               xxx
     * @param readExcelCallBack xxx
     */
    private static void readCell(Row row, ReadExcelCallBack readExcelCallBack) {
        int totalCell = row.getLastCellNum();
        boolean isFirst = true;
        boolean isLast = (totalCell < 2);
        for (int i = 0; i < totalCell; i++) {
            Cell currentCell = row.getCell(i);
            if (currentCell != null) {
                if (isFirst) {
                    readExcelCallBack.readFirstCell(currentCell);
                    isFirst = false;
                } else {
                    isLast = (i == totalCell - 1);
                }
                readExcelCallBack.readCell(currentCell, i);
                if (isLast) {
                    readExcelCallBack.readLastCell(currentCell);
                }
            }
        }
    }

    /**
     * 清理空白行
     * @param sourceExcel xxx
     * @param targetExcel xxx
     * @throws IOException xxx
     * @throws InvalidFormatException xxx
     */
    public static void clearBlankRows(String sourceExcel, String targetExcel) throws IOException, InvalidFormatException {
        readWorkbook(sourceExcel, new AbstractReadExcelCallBack() {
            private DataFormatter formatter = new DataFormatter();
            private int blankCellCount;
            private List<Integer> rowLIndexToBeRemoved = new ArrayList<>();

            @Override
            public void readRow(Row row, int index) {
                if (logger.isDebugEnabled()) {
                    logger.debug("read row: {}", index);
                }
                if (row == null) {
                    rowLIndexToBeRemoved.add(index);
                }
            }

            @Override
            public void readFirstCell(Cell cell) {
                blankCellCount = 0;
            }

            @Override
            public void readCell(Cell cell, int index) {
                if (StringUtil.isEmpty(formatter.formatCellValue(cell))) {
                    blankCellCount++;
                }
            }

            @Override
            public void readRowComplete(Row row) {
                if (row.getLastCellNum() == blankCellCount) {
                    rowLIndexToBeRemoved.add(row.getRowNum());
                }
            }

            @Override
            public void readSheetComplete(Sheet sheet) {
                if (logger.isDebugEnabled()) {
                    logger.debug("sheet: {} read complete", sheet.getSheetName());
                }
                for (int i = rowLIndexToBeRemoved.size() - 1; i > -1; i--) {
                    Row row = sheet.getRow(rowLIndexToBeRemoved.get(i));
                    if (row == null) {
                        if (rowLIndexToBeRemoved.get(i) < sheet.getLastRowNum()) {
                            sheet.shiftRows(rowLIndexToBeRemoved.get(i) + 1, sheet.getLastRowNum(), -1);
                        }
                    } else {
                        try {
                            if (row.getRowNum() < sheet.getLastRowNum()) {
                                sheet.shiftRows(row.getRowNum() + 1, sheet.getLastRowNum(), -1);
                            } else {
                                sheet.removeRow(row);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
                rowLIndexToBeRemoved.clear();
            }

            @Override
            public void readWorkbookComplete(Workbook workbook) throws IOException {
                FileOutputStream fos = null;
                try {
                    fos = new FileOutputStream(targetExcel);
                    workbook.write(fos);
                } finally {
                    if (fos != null) {
                        fos.flush();
                        fos.close();
                    }
                    workbook.close();
                }
            }
        });
    }

    public static void printExcel(String path) throws IOException, InvalidFormatException {
        ExcelUtil.readWorkbook(path, new AbstractReadExcelCallBack() {
            private DataFormatter formatter = new DataFormatter();

            @Override
            public void readCell(Cell cell, int index) {
                System.out.print(formatter.formatCellValue(cell) + "\t");
            }

            @Override
            public void readLastCell(Cell cell) {
                System.out.println();
            }
        });
    }
}
