
package cn.sylinx.redis.springboot.lock;

import java.util.Random;

import cn.sylinx.redis.springboot.RedisCommand;

/**
 * redis分布式锁
 * 
 * @author hanqz
 *
 */
public class RedisDistributedLockImpl extends AbstractDistributedLockImpl {

	private RedisCommand redisCommand;

	/** 加锁标志 */
	private static final String LOCKED = "TRUE";

	/** 毫秒与毫微秒的换算单位 1毫秒 = 1000000毫微秒 */
	private static final long MILLI_NANO_CONVERSION = 1000 * 1000L;

	/** 默认超时时间（毫秒） */
	private static final long DEFAULT_TIME_OUT = 100;

	private static final Random RANDOM = new Random();

	/** 锁的超时时间（秒），过期删除 */
	private static final long EXPIRE = 3 * 60;

	private long expire = 0;

	public void setRedisCommand(RedisCommand redisCommand) {
		this.redisCommand = redisCommand;
	}

	public void setExpire(long expire) {
		this.expire = expire;
	}

	@Override
	public boolean lock(String key) {
		long secondsToRelease = EXPIRE;
		if (expire > 0) {
			secondsToRelease = expire;
		}

		return lock(key, DEFAULT_TIME_OUT, secondsToRelease);
	}

	@Override
	public void unlock(final String key) {
		redisCommand.del(key);
	}

	/**
	 * 加锁 应该以： lock(); try { doSomething(); } finally { unlock()； } 的方式调用
	 * 
	 * @param timeout
	 *            超时时间
	 * @return 成功或失败标志
	 */
	private boolean lock(final String key, final long timeout, final long secondsToRelease) {

		long nano = System.nanoTime();
		long timeoutInner = timeout * MILLI_NANO_CONVERSION;

		try {
			while ((System.nanoTime() - nano) < timeoutInner) {
				if (redisCommand.setnx(key, LOCKED)) {
					redisCommand.expire(key, secondsToRelease);
					return true;
				}
				// 短暂休眠，避免出现活锁
				Thread.sleep(3, RANDOM.nextInt(500));
			}
		} catch (Exception e) {
			throw new RuntimeException("Locking error", e);
		}

		return false;

	}

}