package cn.sylinx.redis.springboot;

import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.data.redis.connection.MessageListener;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.Subscription;
import org.springframework.data.redis.serializer.RedisSerializer;

import cn.sylinx.redis.springboot.serial.SerializerUtil;

public class RedisCommand {

	private static final Log log = LogFactory.getLog(RedisCommand.class);

	private RedisSerializer<String> serializer = SerializerUtil.getStringRedisSerializer();

	private RedisConnectionFactory redisConnectionFactory;

	public void setRedisConnectionFactory(RedisConnectionFactory redisConnectionFactory) {
		this.redisConnectionFactory = redisConnectionFactory;
	}

	public Long publish(final byte[] channel, final byte[] message) {

		return executeCommand((c, o) -> c.publish(channel, message));
	}

	public Long publish(final String channel, final String message) {
		return publish(serializer.serialize(channel), serializer.serialize(message));
	}

	public byte[] get(byte[] key) {
		return executeCommand((c, o) -> c.get(key));
	}

	public String get(String key) {

		byte[] b = get(serializer.serialize(key));
		return serializer.deserialize(b);
	}

	public void set(byte[] key, byte[] value) {

		executeCommand((c, o) -> {
			c.set(key, value);
			return null;
		});
	}

	public void set(String key, String value) {
		set(serializer.serialize(key), serializer.serialize(value));
	}

	public Boolean setnx(final byte[] key, final byte[] value) {
		return executeCommand((c, o) -> c.setNX(key, value));
	}

	public Boolean setnx(final String key, final String value) {
		return setnx(serializer.serialize(key), serializer.serialize(value));
	}

	public void hmset(final byte[] key, final Map<byte[], byte[]> hash) {

		executeCommand((c, o) -> {
			c.hMSet(key, hash);
			return null;
		});
	}

	public void hmset(final String key, final Map<String, String> hash) {
		byte[] bkey = serializer.serialize(key);
		Map<byte[], byte[]> bhash = new HashMap<byte[], byte[]>();
		if (hash != null) {
			for (Map.Entry<String, String> entry : hash.entrySet()) {
				bhash.put(serializer.serialize(entry.getKey()), serializer.serialize(entry.getValue()));
			}
		}
		hmset(bkey, bhash);
	}

	public void del(String... keys) {

		if (keys == null || keys.length < 1) {
			throw new IllegalArgumentException("参数错误");
		}

		byte[][] btkeys = new byte[keys.length][];
		for (int i = 0; i < keys.length; ++i) {
			btkeys[i] = serializer.serialize(keys[i]);
		}
		del(btkeys);
	}

	public void del(byte[]... keys) {
		executeCommand((c, o) -> {
			c.del(keys);
			return null;
		});
	}

	public Boolean expire(String key, long seconds) {
		return expire(serializer.serialize(key), seconds);
	}

	public Boolean expire(byte[] key, long seconds) {
		return executeCommand((c, o) -> c.expire(key, seconds));
	}

	public Set<byte[]> keys(byte[] pattern) {
		return executeCommand((c, o) -> c.keys(pattern));
	}

	public Set<String> keys(String pattern) {
		Set<byte[]> keys = keys(serializer.serialize(pattern));
		Set<String> strKeys = new HashSet<String>();
		for (byte[] key : keys) {
			strKeys.add(serializer.deserialize(key));
		}
		return strKeys;
	}

	public void setex(final byte[] key, final long seconds, final byte[] value) {
		executeCommand((c, o) -> {
			c.setEx(key, seconds, value);
			return null;
		});
	}

	public void setex(final String key, final long seconds, final String value) {
		setex(serializer.serialize(key), seconds, serializer.serialize(value));
	}

	public Boolean exists(final byte[] key) {
		return executeCommand((c, o) -> c.exists(key));
	}

	public Boolean exists(final String key) {
		return exists(serializer.serialize(key));
	}

	public byte[] hget(final byte[] key, final byte[] field) {
		return executeCommand((c, o) -> c.hGet(key, field));
	}

	public String hget(final String key, final String field) {
		byte[] rst = hget(serializer.serialize(key), serializer.serialize(field));
		return serializer.deserialize(rst);
	}

	public Map<byte[], byte[]> hgetAll(final byte[] key) {
		return executeCommand((c, o) -> c.hGetAll(key));
	}

	public Map<String, String> hgetAll(final String key) {
		Map<byte[], byte[]> bmap = hgetAll(serializer.serialize(key));
		Map<String, String> smap = new HashMap<String, String>();
		if (bmap != null) {
			for (Map.Entry<byte[], byte[]> entry : bmap.entrySet()) {
				smap.put(serializer.deserialize(entry.getKey()), serializer.deserialize(entry.getValue()));
			}
		}
		return smap;
	}

	public Boolean hset(final byte[] key, final byte[] field, final byte[] value) {
		return executeCommand((c, o) -> c.hSet(key, field, value));

	}

	public Boolean hset(final String key, final String field, final String value) {
		return hset(serializer.serialize(key), serializer.serialize(field), serializer.serialize(value));
	}

	public void subscribe(MessageListener listener, byte[]... channels) {

		executeCommand((c, o) -> {
			c.subscribe(listener, channels);
			return null;
		});
	}

	public void subscribe(AbstractMessageListener listener, String... channels) {

		byte[][] bchannels = new byte[channels.length][];
		for (int i = 0; i < channels.length; ++i) {
			bchannels[i] = serializer.serialize(channels[i]);
		}
		subscribe(listener, bchannels);
	}

	public void unsubscribe(byte[]... channels) {
		executeCommand((c, o) -> {
			Subscription sub = c.getSubscription();
			if (sub != null) {
				sub.unsubscribe(channels);
			}
			return null;
		});
	}

	public void unsubscribe(String... channels) {
		byte[][] bchannels = new byte[channels.length][];
		for (int i = 0; i < channels.length; ++i) {
			bchannels[i] = serializer.serialize(channels[i]);
		}
		unsubscribe(bchannels);
	}

	public <T> T executeCommand(Executeable<T> exe, Object... objects) {

		RedisConnection connection = getRedisConnection();
		try {
			return exe.cmd(connection, objects);
		} finally {

			if (connection != null) {
				connection.close();
			}
		}
	}

	public static interface Executeable<T> {
		T cmd(RedisConnection connection, Object... objects);
	}

	protected RedisConnection getRedisConnection() {
		int errorCount = 0;

		while (errorCount < 3) {

			try {
				return redisConnectionFactory.getConnection();
			} catch (Exception e) {
				errorCount++;
				e.printStackTrace();
			}
		}

		return null;
	}

	void detect() {

		ExecutorService executorService = Executors.newSingleThreadExecutor();
		executorService.submit((Runnable) () -> {
			while (true) {

				try {
					Thread.sleep(2 * 60 * 1000l);
					get("REDIS-DETECT");
					log.info("redis detect at: " + new Date());

				} catch (Exception e) {

					if (e instanceof InterruptedException) {
						log.info("detect thread interrupted");
						break;
					} else {
						log.error("detect redis error:", e);
					}
				}
			}
		});
		executorService.shutdown();
	}

}
