package cn.sylinx.hbatis.springboot.starter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.bind.RelaxedPropertyResolver;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;

import com.alibaba.druid.pool.DruidDataSource;

import cn.sylinx.hbatis.ext.xmapper.plugin.XmapperPlugin;
import cn.sylinx.hbatis.log.GLog;
import cn.sylinx.hbatis.plugin.IPlugin;
import cn.sylinx.hbatis.plugin.PluginStarter;
import cn.sylinx.hbatis.plugin.datasource.DataSourcePlugin;
import cn.sylinx.hbatis.plugin.ehcache.EhcachePlugin;
import cn.sylinx.hbatis.plugin.model.ModelPreloadPlugin;

@Configuration
public class HbatisStarter implements EnvironmentAware, DisposableBean {

	// 数据源
	private DruidDataSource defaultDataSource;
	private Map<String, DruidDataSource> multiDataSources = new HashMap<String, DruidDataSource>();
	private List<IPlugin> pluginList = new ArrayList<IPlugin>();
	private PluginStarter pluginStarter;

	/**
	 * 创建DataSource
	 *
	 * @param type
	 * @param driverClassName
	 * @param url
	 * @param username
	 * @param password
	 * @return
	 */
	private DruidDataSource buildDataSource(Map<String, Object> dsMap) {

		Object dcnObject = dsMap.get("driver");
		String driverClassName = null;
		if (dcnObject != null && !"".equals(dcnObject)) {
			driverClassName = dcnObject.toString();
		}

		String dbType = dsMap.get("dbtype").toString();

		DruidDataSource cds = new DruidDataSource();
		cds.setUrl(dsMap.get("url").toString());
		cds.setDriverClassName(driverClassName);
		cds.setUsername(dsMap.get("username").toString());
		cds.setPassword(dsMap.get("password").toString());
		cds.setInitialSize(Integer.valueOf(dsMap.get("initalSize").toString()));
		cds.setMinIdle(Integer.valueOf(dsMap.get("minIdle").toString()));
		cds.setMaxActive(Integer.valueOf(dsMap.get("maxActive").toString()));
		cds.setDbType(dbType);
		cds.setTestWhileIdle(true);
		cds.setValidationQuery(ValidateQuery.ME.getValidateQuery(dbType));

		return cds;
	}

	/**
	 * 加载多数据源配置
	 */
	public void setEnvironment(Environment env) {

		initDefaultDataSource(env);
		initMultiDataSources(env);
		initDataSourcePlugin(env);
		initXmapperPlugin(env);
		initModelPreLoadPlugin(env);
		initEhcachePlugin(env);

		initPluginStarter();
	}

	private void initPluginStarter() {
		pluginStarter = new PluginStarter(pluginList);
		pluginStarter.start();
	}

	/**
	 * 初始化数据源plugin
	 * 
	 * @param env
	 */
	private void initDataSourcePlugin(Environment env) {

		// 添加默认数据源
		DataSourcePlugin p = new DataSourcePlugin();
		p.setDataSource(defaultDataSource);
		p.setDbType(defaultDataSource.getDbType());
		pluginList.add(p);

		Set<Entry<String, DruidDataSource>> kvSets = multiDataSources.entrySet();
		for (Entry<String, DruidDataSource> kv : kvSets) {

			DataSourcePlugin ptmp = new DataSourcePlugin();
			ptmp.setDataSource(kv.getValue());
			ptmp.setJdbcResourceName(kv.getKey());
			ptmp.setDbType(kv.getValue().getDbType());
			pluginList.add(ptmp);
		}
	}

	/**
	 * 初始化xmapper插件
	 * 
	 * @param env
	 */
	private void initXmapperPlugin(Environment env) {

		RelaxedPropertyResolver propertyResolver = new RelaxedPropertyResolver(env, "hbatis.plugin.xmapper.");
		String inited = propertyResolver.getProperty("inited");
		if (!"true".equalsIgnoreCase(inited)) {
			return;
		}
		GLog.debug("XmapperPlugin inited.");
		// 初始化
		String resourcePath = propertyResolver.getProperty("resourcePath");
		if (resourcePath == null || "".equals(resourcePath)) {
			resourcePath = "xmapper";
		}
		GLog.debug("XmapperPlugin resourcePath:{}", resourcePath);

		XmapperPlugin xmapperPlugin = new XmapperPlugin(resourcePath);
		pluginList.add(xmapperPlugin);
	}

	/**
	 * 初始化modelpreload插件
	 * 
	 * @param env
	 */
	private void initModelPreLoadPlugin(Environment env) {
		// scanPackageList

		RelaxedPropertyResolver propertyResolver = new RelaxedPropertyResolver(env, "hbatis.plugin.modelpreload.");
		String inited = propertyResolver.getProperty("inited");
		if (!"true".equalsIgnoreCase(inited)) {
			return;
		}
		GLog.debug("ModelPreloadPlugin inited.");
		String scanPackageList = propertyResolver.getProperty("scanPackageList");
		if (scanPackageList == null || "".equals(scanPackageList)) {
			return;
		}
		GLog.debug("ModelPreloadPlugin scanPackageList:{}", scanPackageList);
		ModelPreloadPlugin mp = new ModelPreloadPlugin();
		String[] pkgArray = scanPackageList.split(",");
		mp.setScanPackageList(Arrays.asList(pkgArray));
		pluginList.add(mp);
	}

	/**
	 * 初始化ehcache插件
	 * 
	 * @param env
	 */
	private void initEhcachePlugin(Environment env) {
		RelaxedPropertyResolver propertyResolver = new RelaxedPropertyResolver(env, "hbatis.plugin.ehcache.");
		String inited = propertyResolver.getProperty("inited");
		if (!"true".equalsIgnoreCase(inited)) {
			return;
		}
		GLog.debug("EhcachePlugin inited.");
		pluginList.add(new EhcachePlugin());
	}

	/**
	 * 初始化主数据源
	 */
	private void initDefaultDataSource(Environment env) {
		// 读取主数据源
		RelaxedPropertyResolver propertyResolver = new RelaxedPropertyResolver(env, "hbatis.datasource.default.");
		Map<String, Object> dsMap = new HashMap<String, Object>();

		dsMap.put("url", propertyResolver.getProperty("url"));
		dsMap.put("driver", propertyResolver.getProperty("driver"));
		dsMap.put("username", propertyResolver.getProperty("username"));
		dsMap.put("password", propertyResolver.getProperty("password"));
		dsMap.put("initalSize", propertyResolver.getProperty("initalSize", "2"));
		dsMap.put("minIdle", propertyResolver.getProperty("minIdle", "1"));
		dsMap.put("maxActive", propertyResolver.getProperty("maxActive", "20"));
		dsMap.put("dbtype", propertyResolver.getProperty("dbtype"));
		defaultDataSource = buildDataSource(dsMap);
	}

	/**
	 * 初始化更多数据源
	 */
	private void initMultiDataSources(Environment env) {
		// 读取配置文件获取更多数据源，也可以通过defaultDataSource读取数据库获取更多数据源
		RelaxedPropertyResolver propertyResolver = new RelaxedPropertyResolver(env, "hbatis.datasource.multi.");
		String dsPrefixs = propertyResolver.getProperty("names");
		if (dsPrefixs == null || "".equals(dsPrefixs)) {
			return;
		}

		for (String dsPrefix : dsPrefixs.split(",")) {// 多个数据源
			Map<String, Object> dsMap = propertyResolver.getSubProperties(dsPrefix + ".");
			DruidDataSource ds = buildDataSource(dsMap);
			multiDataSources.put(dsPrefix, ds);
		}
	}

	public void destroy() throws Exception {
		if (pluginStarter != null) {
			pluginStarter.stop();
		}
	}

}
