package cn.sylinx.hbatis.ext.tool;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.sylinx.hbatis.db.mapper.ModelBuilder;
import cn.sylinx.hbatis.db.mapper.anno.BlobType;
import cn.sylinx.hbatis.db.mapper.anno.ColumnDesc;
import cn.sylinx.hbatis.db.mapper.anno.TextType;
import cn.sylinx.hbatis.kit.StrKit;
import cn.sylinx.hbatis.plugin.model.ModelFabric;

public class SqlserverRepositoryUtil {

	public static final String SQLSERVER_QUOTES_L = "[";
	public static final String SQLSERVER_QUOTES_R = "]";
	public static final String SQLSERVER_CREATE_TEMPLATE = "CREATE TABLE [dbo].[%s] (";
	public static final String SQLSERVER_ROW_PK = "CONSTRAINT [PK_%s] PRIMARY KEY CLUSTERED ([id]) ";
	public static final String SQLSERVER_TABLE_COMMENT = "EXEC sp_addextendedproperty 'MS_Description', N'%s', 'SCHEMA', 'dbo', 'TABLE', '%s';";
	public static final String SQLSERVER_COLUMN_COMMENT = "EXEC sp_addextendedproperty 'MS_Description', N'%s', 'SCHEMA', 'dbo', 'TABLE', '%s', 'COLUMN', '%s';";

	public static final String SQLSERVER_ALTER_TABLE = "ALTER TABLE [dbo].[%S] ";
	public static final String SQLSERVER_ALTER_TABLE_ADD_COL = "ADD";
	public static final String SQLSERVER_ALTER_TABLE_MODIFY_COL = "ALTER COLUMN";

	// table, column, comment, table, column, comment, table, column
	public static final String SQLSERVER_ALTER_COLUMN_COMMENT = "IF ((SELECT COUNT(*) FROM ::fn_listextendedproperty('MS_Description', 'SCHEMA', 'dbo', 'TABLE', '%s', 'COLUMN', '%s')) > 0) EXEC sp_updateextendedproperty 'MS_Description', N'%s', 'SCHEMA', 'dbo', 'TABLE', '%s', 'COLUMN', '%s' ELSE EXEC sp_addextendedproperty 'MS_Description', N'%s', 'SCHEMA', 'dbo', 'TABLE', '%s', 'COLUMN', '%s'";

	protected final static Map<String, String> SQLSERVER_JAVA_JDBC_MAP = new HashMap<String, String>();

	static {

		SQLSERVER_JAVA_JDBC_MAP.put("java.util.Date", "datetime");
		SQLSERVER_JAVA_JDBC_MAP.put("java.lang.String", "varchar");
		SQLSERVER_JAVA_JDBC_MAP.put("java.lang.String.MIN", "text");
		SQLSERVER_JAVA_JDBC_MAP.put("java.lang.String.MEDIUM", "text");
		SQLSERVER_JAVA_JDBC_MAP.put("java.lang.String.LONG", "text");
		SQLSERVER_JAVA_JDBC_MAP.put("java.lang.Long", "bigint");
		SQLSERVER_JAVA_JDBC_MAP.put("java.lang.Byte", "tinyint");
		SQLSERVER_JAVA_JDBC_MAP.put("java.lang.Short", "smallint");
		SQLSERVER_JAVA_JDBC_MAP.put("java.math.BigInteger", "bigint");
		SQLSERVER_JAVA_JDBC_MAP.put("java.math.BigDecimal", "decimal");
		SQLSERVER_JAVA_JDBC_MAP.put("java.lang.Integer", "int");
		SQLSERVER_JAVA_JDBC_MAP.put("java.lang.Float", "decimal(12,4)");
		SQLSERVER_JAVA_JDBC_MAP.put("java.lang.Double", "decimal(16,5)");
		SQLSERVER_JAVA_JDBC_MAP.put("java.lang.Boolean", "bit");
		SQLSERVER_JAVA_JDBC_MAP.put("[B", "binary");
		SQLSERVER_JAVA_JDBC_MAP.put("[B.MIN", "binary");
		SQLSERVER_JAVA_JDBC_MAP.put("[B.MEDIUM", "binary");
		SQLSERVER_JAVA_JDBC_MAP.put("[B.LONG", "binary");
	}

	private Class<? extends Object> clz;
	private List<Field> fields;
	private String tableName;
	private String tableDesc;
	private Map<String, String> attrMapping;
	private Map<String, ColumnDesc> fieldDesc;
	private Map<String, Field> fieldMap;

	public SqlserverRepositoryUtil(Class<? extends Object> clz) {

		this.clz = clz;

		if (clz == null) {
			throw new RuntimeException("clz is null or path is null");
		}

		ModelFabric modelFabric = ModelBuilder.getModelFabric(this.clz);
		fields = modelFabric.getFields();
		tableName = modelFabric.getTableName();
		tableDesc = modelFabric.getTable().desc();
		attrMapping = modelFabric.getAttrMapping();
		fieldDesc = modelFabric.getFieldDesc();
		fieldMap = modelFabric.getFieldMap();

	}

	/**
	 * 生成表的SQL
	 * 
	 * @return
	 */
	public String generateSql() {

		StringBuilder commentSb = new StringBuilder();
		StringBuilder sb = new StringBuilder();
		sb.append(String.format(SQLSERVER_CREATE_TEMPLATE, tableName)).append("\n");
		List<Field> sortField = new ArrayList<Field>();
		for (Field field : fields) {
			if (field.getName().equals("id")) {
				sortField.add(0, field);
			} else {
				sortField.add(field);
			}
		}
		for (Field field : sortField) {
			sb.append(generateSingleColumn(field, commentSb)).append("\n");
		}
		sb.append(String.format(SQLSERVER_ROW_PK, tableName)).append(");\n");

		if (StrKit.isNotBlank(tableDesc)) {
			sb.append(String.format(SQLSERVER_TABLE_COMMENT, tableDesc, tableName)).append("\n");
		}

		if (commentSb.length() > 0) {
			sb.append(commentSb.toString());
		}

		return sb.toString();
	}

	private String generateSingleColumn(Field field, StringBuilder commentSb) {

		StringBuilder sb = new StringBuilder();

		String propertyName = field.getName();
		ColumnDesc cd = fieldDesc.get(propertyName);
		String javaType = getFieldType(field);

		String jdbcType = toJdbcType(javaType);
		String columName = attrMapping.get(propertyName);
		if (StrKit.isBlank(columName)) {
			columName = propertyName;
		}

		String column_template = SQLSERVER_QUOTES_L + "%s" + SQLSERVER_QUOTES_R + " %s,";

		int len = 0, precision = 0;
		boolean blob = false, text = false;
		BlobType bt = BlobType.MIN;
		TextType tt = TextType.MIN;

		String comment = null;
		boolean nullable = true;

		if (cd != null) {
			len = cd.len();
			precision = cd.precision();
			comment = cd.desc();
			nullable = cd.nullable();
			blob = cd.blob();
			text = cd.text();
			bt = cd.blobType();
			tt = cd.textType();
		}

		if (blob) {
			// 二进制判断
			String key = javaType + "." + bt.name();
			jdbcType = toJdbcType(key);
		} else if (text) {
			// 文本
			String key = javaType + "." + tt.name();
			jdbcType = toJdbcType(key);
		} else if (len > 0) {
			jdbcType = jdbcType + (precision == 0 ? "(" + len + ")" : "(" + len + "," + precision + ")");
		} else if ("varchar".equals(jdbcType)) {
			jdbcType = jdbcType + "(50)";
		}

		// 空判断
		jdbcType = jdbcType + (nullable ? " NULL" : " NOT NULL");

		if ("id".equals(field.getName())) {
			// id主键
			jdbcType = "bigint IDENTITY(1,1) NOT NULL";
		}

		sb.append(String.format(column_template, columName, jdbcType));

		if (StrKit.isNotBlank(comment)) {
			commentSb.append(String.format(SQLSERVER_COLUMN_COMMENT, comment, tableName, columName)).append("\n");
		}

		return sb.toString();
	}

	/**
	 * 获取javaType对应的 jdbcType
	 * 
	 * @param javaType
	 * @return
	 */
	protected String toJdbcType(String javaType) {

		return SQLSERVER_JAVA_JDBC_MAP.get(javaType);
	}

	/**
	 * 获取字段类型
	 * 
	 * @param field
	 * @return
	 */
	protected String getFieldType(Field field) {

		return field.getType().getName();
	}

	public List<Field> getFields() {
		return fields;
	}

	public String getTableName() {
		return tableName;
	}

	public void setTableName(String tableName) {
		this.tableName = tableName;
	}

	public String generateModifyTableColumnDDL(String prop) {

		StringBuilder sb = new StringBuilder();
		sb.append(String.format(SQLSERVER_ALTER_TABLE, tableName)).append(" ").append(SQLSERVER_ALTER_TABLE_MODIFY_COL)
				.append(" ");
		String propertyName = prop;
		Field field = fieldMap.get(prop);
		ColumnDesc cd = fieldDesc.get(propertyName);
		String javaType = getFieldType(field);

		String jdbcType = toJdbcType(javaType);
		String columName = attrMapping.get(propertyName);
		if (StrKit.isBlank(columName)) {
			columName = propertyName;
		}

		String column_template = SQLSERVER_QUOTES_L + "%s" + SQLSERVER_QUOTES_R + " %s;\n";

		int len = 0, precision = 0;
		boolean blob = false, text = false;
		BlobType bt = BlobType.MIN;
		TextType tt = TextType.MIN;

		String comment = null;
		boolean nullable = true;

		if (cd != null) {
			len = cd.len();
			precision = cd.precision();
			comment = cd.desc();
			nullable = cd.nullable();
			blob = cd.blob();
			text = cd.text();
			bt = cd.blobType();
			tt = cd.textType();
		}

		if (blob) {
			// 二进制判断
			String key = javaType + "." + bt.name();
			jdbcType = toJdbcType(key);
		} else if (text) {
			// 文本
			String key = javaType + "." + tt.name();
			jdbcType = toJdbcType(key);
		} else if (len > 0) {
			jdbcType = jdbcType + (precision == 0 ? "(" + len + ")" : "(" + len + "," + precision + ")");
		} else if ("varchar".equals(jdbcType)) {
			jdbcType = jdbcType + "(50)";
		}

		// 空判断
		jdbcType = jdbcType + (nullable ? " NULL" : " NOT NULL");

		if ("id".equals(field.getName())) {
			// id主键
			jdbcType = "bigint IDENTITY(1,1) NOT NULL";
		}

		sb.append(String.format(column_template, columName, jdbcType));

		if (StrKit.isNotBlank(comment)) {
			sb.append(createColumnCommentSql(columName, comment));
		}

		return sb.toString();
	}

	private String createColumnCommentSql(String column, String comment) {
		// table, column, comment, table, column, comment, table, column
		String cmt = String.format(SQLSERVER_ALTER_COLUMN_COMMENT, tableName, column, comment, tableName, column,
				comment, tableName, column);
		return cmt + ";\n";
	}

	public String generateAddTableColumnDDL(String prop) {

		StringBuilder sb = new StringBuilder();
		sb.append(String.format(SQLSERVER_ALTER_TABLE, tableName)).append(" ").append(SQLSERVER_ALTER_TABLE_ADD_COL)
				.append(" ");
		String propertyName = prop;
		Field field = fieldMap.get(prop);
		ColumnDesc cd = fieldDesc.get(propertyName);
		String javaType = getFieldType(field);

		String jdbcType = toJdbcType(javaType);
		String columName = attrMapping.get(propertyName);
		if (StrKit.isBlank(columName)) {
			columName = propertyName;
		}

		String column_template = SQLSERVER_QUOTES_L + "%s" + SQLSERVER_QUOTES_R + " %s;\n";

		int len = 0, precision = 0;
		boolean blob = false, text = false;
		BlobType bt = BlobType.MIN;
		TextType tt = TextType.MIN;

		String comment = null;
		boolean nullable = true;

		if (cd != null) {
			len = cd.len();
			precision = cd.precision();
			comment = cd.desc();
			nullable = cd.nullable();
			blob = cd.blob();
			text = cd.text();
			bt = cd.blobType();
			tt = cd.textType();
		}

		if (blob) {
			// 二进制判断
			String key = javaType + "." + bt.name();
			jdbcType = toJdbcType(key);
		} else if (text) {
			// 文本
			String key = javaType + "." + tt.name();
			jdbcType = toJdbcType(key);
		} else if (len > 0) {
			jdbcType = jdbcType + (precision == 0 ? "(" + len + ")" : "(" + len + "," + precision + ")");
		} else if ("varchar".equals(jdbcType)) {
			jdbcType = jdbcType + "(50)";
		}

		// 空判断
		jdbcType = jdbcType + (nullable ? " NULL" : " NOT NULL");

		if ("id".equals(field.getName())) {
			// id主键
			jdbcType = "bigint IDENTITY(1,1) NOT NULL";
		}

		sb.append(String.format(column_template, columName, jdbcType));

		if (StrKit.isNotBlank(comment)) {
			sb.append(String.format(SQLSERVER_COLUMN_COMMENT, comment, tableName, columName)).append("\n");
		}

		return sb.toString();
	}
}