package cn.sylinx.hbatis.ext.tool;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cn.sylinx.hbatis.db.mapper.ModelBuilder;
import cn.sylinx.hbatis.db.mapper.anno.BlobType;
import cn.sylinx.hbatis.db.mapper.anno.ColumnDesc;
import cn.sylinx.hbatis.db.mapper.anno.TextType;
import cn.sylinx.hbatis.kit.StrKit;
import cn.sylinx.hbatis.plugin.model.ModelFabric;

public class OracleRepositoryUtil {

	public static final String ORACLE_SINGLE_QUOTES = "\"";

	public static final String ORACLE_DROP_TEMPLATE = "DROP TABLE IF EXISTS \"%s\";";

	public static final String ORACLE_CREATE_TEMPLATE = "CREATE TABLE \"%s\" (";

	public static final String ORACLE_ROW_PK = " CONSTRAINT \"P_%s\" PRIMARY KEY(\"ID\") ";

	public static final String ORACLE_ALTER_TABLE = "ALTER TABLE \"%s\"";

	public static final String ORACLE_ALTER_TABLE_ADD_COL = "ADD";

	public static final String ORACLE_ALTER_TABLE_MODIFY_COL = "MODIFY";

	protected final static Map<String, String> ORACLE_JAVA_JDBC_MAP = new HashMap<String, String>();

	static {

		ORACLE_JAVA_JDBC_MAP.put("java.util.Date", "DATE");
		ORACLE_JAVA_JDBC_MAP.put("java.lang.String", "VARCHAR2");
		ORACLE_JAVA_JDBC_MAP.put("java.lang.String.MIN", "CLOB");
		ORACLE_JAVA_JDBC_MAP.put("java.lang.String.MEDIUM", "CLOB");
		ORACLE_JAVA_JDBC_MAP.put("java.lang.String.LONG", "CLOB");
		ORACLE_JAVA_JDBC_MAP.put("java.lang.Long", "NUMBER(20)");
		ORACLE_JAVA_JDBC_MAP.put("java.lang.Byte", "NUMBER(3,0)");
		ORACLE_JAVA_JDBC_MAP.put("java.lang.Short", "NUMBER(6,0)");
		ORACLE_JAVA_JDBC_MAP.put("java.math.BigInteger", "NUMBER(20)");
		ORACLE_JAVA_JDBC_MAP.put("java.math.BigDecimal", "NUMBER");
		ORACLE_JAVA_JDBC_MAP.put("java.lang.Integer", "NUMBER(10,0)");
		ORACLE_JAVA_JDBC_MAP.put("java.lang.Float", "NUMBER(12,3)");
		ORACLE_JAVA_JDBC_MAP.put("java.lang.Double", "NUMBER(16,3)");
		ORACLE_JAVA_JDBC_MAP.put("java.lang.Boolean", "NUMBER(1,0)");
		ORACLE_JAVA_JDBC_MAP.put("[B", "BLOB");
		ORACLE_JAVA_JDBC_MAP.put("[B.MIN", "BLOB");
		ORACLE_JAVA_JDBC_MAP.put("[B.MEDIUM", "BLOB");
		ORACLE_JAVA_JDBC_MAP.put("[B.LONG", "BLOB");
	}

	private Class<? extends Object> clz;
	private List<Field> fields;
	private String tableName;
	private String tableDesc;
	private Map<String, String> attrMapping;
	private Map<String, ColumnDesc> fieldDesc;
	private Map<String, Field> fieldMap;

	public OracleRepositoryUtil(Class<? extends Object> clz) {

		this.clz = clz;

		if (clz == null) {
			throw new RuntimeException("clz is null or path is null");
		}

		ModelFabric modelFabric = ModelBuilder.getModelFabric(this.clz);
		fields = modelFabric.getFields();
		tableName = modelFabric.getTableName().toUpperCase();
		tableDesc = modelFabric.getTable().desc();
		attrMapping = modelFabric.getAttrMapping();
		fieldDesc = modelFabric.getFieldDesc();
		fieldMap = modelFabric.getFieldMap();

	}

	public String generateAddTableColumnDDL(String prop) {

		StringBuilder sb = new StringBuilder();
		sb.append(String.format(ORACLE_ALTER_TABLE, tableName)).append(" ").append(ORACLE_ALTER_TABLE_ADD_COL)
				.append(" ");
		String propertyName = prop;
		Field field = fieldMap.get(prop);
		ColumnDesc cd = fieldDesc.get(propertyName);
		String javaType = getFieldType(field);

		String jdbcType = toJdbcType(javaType);
		String columName = attrMapping.get(propertyName);
		if (StrKit.isBlank(columName)) {
			columName = propertyName;
		}
		columName = columName.toUpperCase();

		String column_template = ORACLE_SINGLE_QUOTES + "%s" + ORACLE_SINGLE_QUOTES + " %s;\n";

		int len = 0, precision = 0;
		boolean blob = false, text = false;
		BlobType bt = BlobType.MIN;
		TextType tt = TextType.MIN;

		String comment = null;
		boolean nullable = true;

		if (cd != null) {
			len = cd.len();
			precision = cd.precision();
			comment = cd.desc();
			nullable = cd.nullable();
			blob = cd.blob();
			text = cd.text();
			bt = cd.blobType();
			tt = cd.textType();
		}

		if (blob) {
			// 二进制判断
			String key = javaType + "." + bt.name();
			jdbcType = toJdbcType(key);
		} else if (text) {
			// 文本
			String key = javaType + "." + tt.name();
			jdbcType = toJdbcType(key);
		} else if (len > 0) {
			jdbcType = jdbcType + (precision == 0 ? "(" + len + ")" : "(" + len + "," + precision + ")");
		} else if ("VARCHAR2".equals(jdbcType)) {
			jdbcType = jdbcType + "(50)";
		}

		// 空判断
		jdbcType = jdbcType + (nullable ? " DEFAULT NULL" : " NOT NULL");

		if ("id".equalsIgnoreCase(propertyName)) {
			jdbcType = " NUMBER NOT NULL";
		}

		sb.append(String.format(column_template, columName, jdbcType));

		// COMMENT
		if (StrKit.isNotBlank(comment)) {
			sb.append("COMMENT ON COLUMN \"").append(tableName).append("\".\"").append(columName).append("\" IS '")
					.append(comment).append("';\n");
		}

		return sb.toString();
	}

	public String generateModifyTableColumnDDL(String prop) {

		if ("id".equalsIgnoreCase(prop)) {
			return null;
		}
		StringBuilder sb = new StringBuilder();
		sb.append(String.format(ORACLE_ALTER_TABLE, tableName)).append(" ").append(ORACLE_ALTER_TABLE_MODIFY_COL)
				.append(" ");
		String propertyName = prop;
		Field field = fieldMap.get(prop);
		ColumnDesc cd = fieldDesc.get(propertyName);
		String javaType = getFieldType(field);

		String jdbcType = toJdbcType(javaType);
		String columName = attrMapping.get(propertyName);
		if (StrKit.isBlank(columName)) {
			columName = propertyName;
		}
		columName = columName.toUpperCase();

		String column_template = ORACLE_SINGLE_QUOTES + "%s" + ORACLE_SINGLE_QUOTES + " %s;\n";

		int len = 0, precision = 0;
		boolean blob = false, text = false;
		BlobType bt = BlobType.MIN;
		TextType tt = TextType.MIN;

		String comment = null;

		if (cd != null) {
			len = cd.len();
			precision = cd.precision();
			comment = cd.desc();
			blob = cd.blob();
			text = cd.text();
			bt = cd.blobType();
			tt = cd.textType();
		}

		if (blob) {
			// 二进制判断
			String key = javaType + "." + bt.name();
			jdbcType = toJdbcType(key);
		} else if (text) {
			// 文本
			String key = javaType + "." + tt.name();
			jdbcType = toJdbcType(key);
		} else if (len > 0) {
			jdbcType = jdbcType + (precision == 0 ? "(" + len + ")" : "(" + len + "," + precision + ")");
		} else if ("VARCHAR2".equals(jdbcType)) {
			jdbcType = jdbcType + "(50)";
		}

		sb.append(String.format(column_template, columName, jdbcType));

		// COMMENT
		if (StrKit.isNotBlank(comment)) {
			sb.append("COMMENT ON COLUMN \"").append(tableName).append("\".\"").append(columName).append("\" IS '")
					.append(comment).append("';\n");
		}

		return sb.toString();
	}

	/**
	 * 生成表的SQL
	 * 
	 * @return
	 */
	public String generateSql() {

		StringBuilder commentSb = new StringBuilder();

		StringBuilder sb = new StringBuilder();
		sb.append(String.format(ORACLE_CREATE_TEMPLATE, tableName)).append("\n");
		List<Field> sortField = new ArrayList<Field>();
		for (Field field : fields) {
			if (field.getName().equals("id")) {
				sortField.add(0, field);
			} else {
				sortField.add(field);
			}
		}
		for (Field field : sortField) {
			sb.append(generateSingleColumn(field, commentSb)).append("\n");
		}
		sb.append(String.format(ORACLE_ROW_PK, tableName)).append("\n");
		sb.append(");\n");

		if (StrKit.isNotBlank(tableDesc)) {
			commentSb.append("COMMENT ON TABLE \"" + tableName + "\" IS '").append(tableDesc).append("';\n");
		}
		sb.append(commentSb);

		return sb.toString();
	}

	private String generateSingleColumn(Field field, StringBuilder commentSb) {

		StringBuilder sb = new StringBuilder();

		String propertyName = field.getName();
		ColumnDesc cd = fieldDesc.get(propertyName);
		String javaType = getFieldType(field);

		String jdbcType = toJdbcType(javaType);
		String columName = attrMapping.get(propertyName);
		if (StrKit.isBlank(columName)) {
			columName = propertyName;
		}
		columName = columName.toUpperCase();

		String column_template = ORACLE_SINGLE_QUOTES + "%s" + ORACLE_SINGLE_QUOTES + " %s,";

		int len = 0, precision = 0;
		boolean blob = false, text = false;
		BlobType bt = BlobType.MIN;
		TextType tt = TextType.MIN;

		String comment = null;
		boolean nullable = true;

		if (cd != null) {
			len = cd.len();
			precision = cd.precision();
			comment = cd.desc();
			nullable = cd.nullable();
			blob = cd.blob();
			text = cd.text();
			bt = cd.blobType();
			tt = cd.textType();
		}

		if (blob) {
			// 二进制判断
			String key = javaType + "." + bt.name();
			jdbcType = toJdbcType(key);
		} else if (text) {
			// 文本
			String key = javaType + "." + tt.name();
			jdbcType = toJdbcType(key);
		} else if (len > 0) {
			jdbcType = jdbcType + (precision == 0 ? "(" + len + ")" : "(" + len + "," + precision + ")");
		} else if ("VARCHAR2".equals(jdbcType)) {
			jdbcType = jdbcType + "(50)";
		}

		// 空判断
		jdbcType = jdbcType + (nullable ? " DEFAULT NULL" : " NOT NULL");

		// COMMENT
		if (StrKit.isNotBlank(comment)) {
			commentSb.append("COMMENT ON COLUMN \"").append(tableName).append("\".\"").append(columName)
					.append("\" IS '").append(comment).append("';\n");
		}

		if ("id".equalsIgnoreCase(columName)) {
			jdbcType = " NUMBER NOT NULL";
		}

		sb.append(String.format(column_template, columName, jdbcType));

		return sb.toString();
	}

	/**
	 * 获取javaType对应的 jdbcType
	 * 
	 * @param javaType
	 * @return
	 */
	protected String toJdbcType(String javaType) {

		return ORACLE_JAVA_JDBC_MAP.get(javaType);
	}

	/**
	 * 获取字段类型
	 * 
	 * @param field
	 * @return
	 */
	protected String getFieldType(Field field) {

		return field.getType().getName();
	}

	public List<Field> getFields() {
		return fields;
	}

	public String getTableName() {
		return tableName;
	}

	public void setTableName(String tableName) {
		this.tableName = tableName;
	}
}