package cn.sylinx.hbatis.ext.starter.common;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;

import cn.sylinx.hbatis.ds.DataSourceWrapper;
import cn.sylinx.hbatis.exception.HbatisException;
import cn.sylinx.hbatis.ext.mirage.plugin.SqlResourcePreloadPlugin;
import cn.sylinx.hbatis.ext.starter.pool.DataSourceWrapperFactory;
import cn.sylinx.hbatis.ext.xmapper.plugin.XmapperPlugin;
import cn.sylinx.hbatis.kit.StrKit;
import cn.sylinx.hbatis.log.GLog;
import cn.sylinx.hbatis.plugin.IPlugin;
import cn.sylinx.hbatis.plugin.PluginStarter;
import cn.sylinx.hbatis.plugin.datasource.DataSourcePlugin;
import cn.sylinx.hbatis.plugin.ehcache.EhcachePlugin;
import cn.sylinx.hbatis.plugin.model.ModelPreloadPlugin;
import cn.sylinx.hbatis.plugin.proxy.ProxyPlugin;
import cn.sylinx.hbatis.plugin.statement.StatementHandlerPlugin;
import cn.sylinx.hbatis.plugin.transaction.TransactionIsolationPlugin;

class StarterBean {

	private String config = "/hbatis.properties";
	private Properties p;

	// 数据源
	private DataSourceWrapper defaultDataSource;

	private Map<String, DataSourceWrapper> multiDataSources = new HashMap<String, DataSourceWrapper>();
	private List<IPlugin> pluginList = new ArrayList<IPlugin>();
	private PluginStarter pluginStarter;

	public StarterBean() {
		this(null);
	}

	public StarterBean(String config) {
		if (config != null && !"".equals(config)) {
			this.config = config;
		}
		init();
	}

	private void init() {

		p = new Properties();
		try {
			p.load(StarterBean.class.getResourceAsStream(config));
		} catch (IOException e) {
			// 配置文件未找到
			throw new HbatisException(e);
		}
	}

	private void initHbatis() {
		initDefaultDataSource(p);
		initMultiDataSources(p);
		initDataSourcePlugin(p);
		initXmapperPlugin(p);
		initModelPreLoadPlugin(p);
		initEhcachePlugin(p);
		initSqlResourcePreloadPlugin(p);
		initStatementHandlerPlugin(p);
		initTransactionPlugin(p);
		initProxyPlugin(p);
	}

	public void addPlugin(IPlugin plugin) {
		pluginList.add(plugin);
	}

	public void start() {

		String initKey = "hbatis.init";
		String initStr = p.getProperty(initKey, "false");
		boolean init = Boolean.valueOf(initStr);
		if (!init) {
			GLog.info("hbatis 没有被初始化，请确认hbatis.init=true。");
			return;
		}

		initHbatis();
		initPluginStarter();
	}

	private void initPluginStarter() {
		pluginStarter = new PluginStarter(pluginList);
		pluginStarter.start();
	}

	/**
	 * 初始化数据源plugin
	 * 
	 * @param env
	 */
	private void initDataSourcePlugin(Properties p) {

		// 添加默认数据源
		DataSourcePlugin dsp = new DataSourcePlugin();
		dsp.setDataSourceWrapper(defaultDataSource);

		addPlugin(dsp);

		Set<Entry<String, DataSourceWrapper>> kvSets = multiDataSources.entrySet();
		for (Entry<String, DataSourceWrapper> kv : kvSets) {

			DataSourcePlugin ptmp = new DataSourcePlugin();
			ptmp.setJdbcResourceName(kv.getKey());
			ptmp.setDataSourceWrapper(kv.getValue());

			addPlugin(ptmp);
		}
	}

	/**
	 * 初始化xmapper插件
	 * 
	 * @param env
	 */
	private void initXmapperPlugin(Properties p) {

		RelaxedPropertyResolver propertyResolver = new RelaxedPropertyResolver(p, "hbatis.plugin.xmapper.");
		String inited = propertyResolver.getProperty("inited");
		if (!"true".equalsIgnoreCase(inited)) {
			return;
		}
		GLog.debug("XmapperPlugin inited.");
		// 初始化
		String resourcePath = propertyResolver.getProperty("resourcePath");
		if (resourcePath == null || "".equals(resourcePath)) {
			resourcePath = "xmapper";
		}
		GLog.debug("XmapperPlugin resourcePath:{}", resourcePath);

		XmapperPlugin xmapperPlugin = new XmapperPlugin(resourcePath);
		addPlugin(xmapperPlugin);
	}

	/**
	 * 初始化modelpreload插件
	 * 
	 * @param env
	 */
	private void initModelPreLoadPlugin(Properties p) {
		// scanPackageList

		RelaxedPropertyResolver propertyResolver = new RelaxedPropertyResolver(p, "hbatis.plugin.modelpreload.");
		String inited = propertyResolver.getProperty("inited");
		if (!"true".equalsIgnoreCase(inited)) {
			return;
		}
		GLog.debug("ModelPreloadPlugin inited.");

		ModelPreloadPlugin mp = new ModelPreloadPlugin();

		String acmStrategy = propertyResolver.getProperty("acmStrategy");
		if (StrKit.isNotBlank(acmStrategy)) {
			// 设置全局模型字段映射策略
			mp.setAcmStrategyClass(acmStrategy);
		}

		String scanPackageList = propertyResolver.getProperty("scanPackageList");
		if (StrKit.isNotBlank(scanPackageList)) {
			GLog.debug("ModelPreloadPlugin scanPackageList:{}", scanPackageList);
			String[] pkgArray = scanPackageList.split(",");
			mp.setScanPackageList(Arrays.asList(pkgArray));
		}

		addPlugin(mp);
	}

	private void initSqlResourcePreloadPlugin(Properties p) {
		// SqlResourcePreloadPlugin
		String path = p.getProperty("hbatis.plugin.mirage.sqlpath");
		if (StrKit.isBlank(path)) {
			return;
		}

		String[] pathArray = path.split(",");
		List<String> pathList = Arrays.asList(pathArray);
		SqlResourcePreloadPlugin plugin = new SqlResourcePreloadPlugin(pathList);
		addPlugin(plugin);
	}
	
	/**
	 * 添加sql语句处理插件
	 * @param p
	 */
	private void initStatementHandlerPlugin(Properties p) {
		String statementHandlerClass = p.getProperty("hbatis.plugin.statement.clazz");
		if(StrKit.isBlank(statementHandlerClass)) {
			return;
		}
		
		StatementHandlerPlugin shp = new StatementHandlerPlugin();
		shp.setStatementHandlerClass(statementHandlerClass);
		addPlugin(shp);
	}

	private void initTransactionPlugin(Properties p) {
		// TransactionIsolationPlugin
		String level = p.getProperty("hbatis.plugin.transaction.level");
		if (StrKit.isBlank(level)) {
			return;
		}

		int levelInt = Integer.parseInt(level);
		TransactionIsolationPlugin plugin = new TransactionIsolationPlugin(levelInt);
		addPlugin(plugin);
	}

	// 初始化代理插件
	private void initProxyPlugin(Properties p) {

		String clzList = p.getProperty("hbatis.plugin.proxy.class");
		if (StrKit.isBlank(clzList)) {
			return;
		}

		ProxyPlugin plugin = new ProxyPlugin();
		String[] clzArray = clzList.split(",");
		for (String clz : clzArray) {
			if (StrKit.isNotBlank(clz)) {
				IPlugin item = null;
				try {
					item = (IPlugin) Class.forName(clz.trim()).newInstance();
				} catch (InstantiationException | IllegalAccessException | ClassNotFoundException e) {
					GLog.error("初始化插件失败, clz: " + clz, e);
				}
				if (item != null) {
					// 加入代理插件
					plugin.addPlugin(item);
				}
			}
		}

		addPlugin(plugin);
	}

	/**
	 * 初始化ehcache插件
	 * 
	 * @param env
	 */
	private void initEhcachePlugin(Properties p) {
		RelaxedPropertyResolver propertyResolver = new RelaxedPropertyResolver(p, "hbatis.plugin.ehcache.");
		String inited = propertyResolver.getProperty("inited");
		if (!"true".equalsIgnoreCase(inited)) {
			return;
		}
		GLog.debug("EhcachePlugin inited.");
		addPlugin(new EhcachePlugin());
	}

	/**
	 * 初始化主数据源
	 */
	private void initDefaultDataSource(Properties p) {
		// 读取主数据源
		RelaxedPropertyResolver propertyResolver = new RelaxedPropertyResolver(p, "hbatis.datasource.default.");
		Map<String, Object> dsMap = new HashMap<String, Object>();

		dsMap.put("url", propertyResolver.getProperty("url"));
		dsMap.put("driver", propertyResolver.getProperty("driver"));
		dsMap.put("username", propertyResolver.getProperty("username"));
		dsMap.put("password", propertyResolver.getProperty("password"));
		dsMap.put("dbtype", propertyResolver.getProperty("dbtype"));

		// 获取其它配置
		Map<String, Object> extConfig = propertyResolver.getSubProperties("config.");

		defaultDataSource = buildDataSource(dsMap, extConfig);
	}

	/**
	 * 初始化更多数据源
	 */
	private void initMultiDataSources(Properties p) {
		// 读取配置文件获取更多数据源，也可以通过defaultDataSource读取数据库获取更多数据源
		RelaxedPropertyResolver propertyResolver = new RelaxedPropertyResolver(p, "hbatis.datasource.multi.");
		String dsPrefixs = propertyResolver.getProperty("names");
		if (dsPrefixs == null || "".equals(dsPrefixs)) {
			return;
		}

		for (String dsPrefix : dsPrefixs.split(",")) {// 多个数据源
			Map<String, Object> dsMap = propertyResolver.getSubProperties(dsPrefix + ".");
			Map<String, Object> extConfig = propertyResolver.getSubProperties(dsPrefix + ".config.");
			DataSourceWrapper ds = buildDataSource(dsMap, extConfig);
			multiDataSources.put(dsPrefix, ds);
		}
	}

	private DataSourceWrapper buildDataSource(Map<String, Object> dsMap, Map<String, Object> extConfig) {
		return DataSourceWrapperFactory.buildDataSource(dsMap, extConfig);
	}

}
