package cn.sylinx.hbatis.io;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.util.List;
import java.util.Properties;

/**
 * 资源常用类
 * 
 * @author han
 *
 */
public class Resources {

	private static ClassLoaderWrapper classLoaderWrapper = new ClassLoaderWrapper();

	private static Charset charset;

	Resources() {
	}

	/**
	 * get the default classloader
	 * 
	 * @return the classloader
	 */
	public static ClassLoader getDefaultClassLoader() {
		return classLoaderWrapper.defaultClassLoader;
	}

	/**
	 * Sets the default classloader
	 *
	 * @param defaultClassLoader
	 *            - the new default ClassLoader
	 */
	public static void setDefaultClassLoader(ClassLoader defaultClassLoader) {
		classLoaderWrapper.defaultClassLoader = defaultClassLoader;
	}

	/**
	 * Returns the URL of the resource on the classpath
	 *
	 * @param resource
	 *            The resource to find
	 * 
	 * @return The resource
	 * 
	 * @throws java.io.IOException
	 *             If the resource cannot be found or read
	 */
	public static URL getResourceURL(String resource) throws IOException {
		return getResourceURL(null, resource);
	}

	public static List<URL> getResourceURLs(String resource) throws IOException {
		return getResourceURLs(null, resource);
	}

	/**
	 * Returns the URL of the resource on the classpath
	 * 
	 * @param loader
	 *            classloader
	 * @param resource
	 *            The resource to find
	 * @return The resource
	 * @throws java.io.IOException
	 *             If the resource cannot be found or read
	 */
	public static URL getResourceURL(ClassLoader loader, String resource) throws IOException {
		URL url = classLoaderWrapper.getResourceAsURL(resource, loader);
		if (url == null) {
			throw new IOException("Could not find resource " + resource);
		}
		return url;
	}

	public static List<URL> getResourceURLs(ClassLoader loader, String resource) throws IOException {
		List<URL> urlList = classLoaderWrapper.getResourceAsURLs(resource, loader);
		if (urlList == null || urlList.isEmpty()) {
			throw new IOException("Could not find resource " + resource);
		}
		return urlList;
	}

	/**
	 * Returns the InputStream of the resource on the classpath
	 * 
	 * @param resource
	 *            the resource to find
	 * @return the resource
	 * @throws IOException
	 *             If the resource cannot be found or read
	 */
	public static InputStream getResourceAsStream(String resource) throws IOException {
		return getResourceAsStream(null, resource);
	}

	/**
	 * returns the InputStream of the resource on the classpath
	 * 
	 * @param loader
	 *            The Classloader
	 * @param resource
	 *            The resource to find
	 * @return the resource
	 * @throws IOException
	 *             If the resource cannot be found or read
	 */
	public static InputStream getResourceAsStream(ClassLoader loader, String resource) throws IOException {
		InputStream in = classLoaderWrapper.getResourceAsStream(resource, loader);
		if (in == null) {
			throw new IOException("Could not find resource " + resource);
		}
		return in;
	}

	/**
	 * return the Properties of the resource on the classpath
	 * 
	 * @param resource
	 *            the resource to find
	 * @return the resource
	 * @throws IOException
	 *             If the resource cannot be found or read
	 */
	public static Properties getResourceAsProperties(String resource) throws IOException {
		Properties props = new Properties();
		InputStream in = getResourceAsStream(resource);
		props.load(in);
		in.close();
		return props;
	}

	/**
	 * return the Properties of the resource on the classpath
	 * 
	 * @param loader
	 *            classloader
	 * @param resource
	 *            the resource to find
	 * @return the resource
	 * @throws IOException
	 *             If the resource cannot be found or read
	 */
	public static Properties getResourceAsProperties(ClassLoader loader, String resource) throws IOException {
		Properties props = new Properties();
		InputStream in = getResourceAsStream(loader, resource);
		props.load(in);
		in.close();
		return props;
	}

	/**
	 * return the Reader of the resource on the classpath
	 * 
	 * @param resource
	 *            the resource to find
	 * @return the resource
	 * @throws IOException
	 *             If the resource cannot be found or read
	 */
	public static Reader getResourceAsReader(String resource) throws IOException {
		Reader reader;
		if (charset == null) {
			reader = new InputStreamReader(getResourceAsStream(resource));
		} else {
			reader = new InputStreamReader(getResourceAsStream(resource), charset);
		}
		return reader;
	}

	/**
	 * return the Reader of the resource on the classpath
	 * 
	 * @param loader
	 *            ClassLoader
	 * @param resource
	 *            the resource to find
	 * @return the resource
	 * @throws IOException
	 *             If the resource cannot be found or read
	 */
	public static Reader getResourceAsReader(ClassLoader loader, String resource) throws IOException {
		Reader reader;
		if (charset == null) {
			reader = new InputStreamReader(getResourceAsStream(loader, resource));
		} else {
			reader = new InputStreamReader(getResourceAsStream(loader, resource), charset);
		}
		return reader;
	}

	/**
	 * return the File of the resource on the classpath
	 * 
	 * @param resource
	 *            the resource to find
	 * @return the resource
	 * @throws IOException
	 *             If the resource cannot be found or read
	 */
	public static File getResourceAsFile(String resource) throws IOException {
		return new File(getResourceURL(resource).getFile());
	}

	/**
	 * return the File of the resource on the classpath
	 * 
	 * @param loader
	 *            ClassLoader
	 * @param resource
	 *            the resource to find
	 * @return the resource
	 * @throws IOException
	 *             If the resource cannot be found or read
	 */
	public static File getResourceAsFile(ClassLoader loader, String resource) throws IOException {
		return new File(getResourceURL(loader, resource).getFile());
	}

	/**
	 * return the InputStream of the url
	 * 
	 * @param urlString
	 *            url string
	 * @return the InputStream
	 * @throws IOException
	 *             If the url cannot be found or read
	 */
	public static InputStream getUrlAsStream(String urlString) throws IOException {
		URL url = new URL(urlString);
		URLConnection conn = url.openConnection();
		return conn.getInputStream();
	}

	/**
	 * return the Reader of the url
	 * 
	 * @param urlString
	 *            url String
	 * @return the Reader of url
	 * @throws IOException
	 *             If the url cannot be found or read
	 */
	public static Reader getUrlAsReader(String urlString) throws IOException {
		Reader reader;
		if (charset == null) {
			reader = new InputStreamReader(getUrlAsStream(urlString));
		} else {
			reader = new InputStreamReader(getUrlAsStream(urlString), charset);
		}
		return reader;
	}

	/**
	 * return the Properties of url
	 * 
	 * @param urlString
	 *            url string
	 * @return the Properties
	 * @throws IOException
	 *             If the url cannot be found or read
	 */
	public static Properties getUrlAsProperties(String urlString) throws IOException {
		Properties props = new Properties();
		InputStream in = getUrlAsStream(urlString);
		props.load(in);
		in.close();
		return props;
	}

	/**
	 * return the Class of the classname
	 * 
	 * @param className
	 *            the classname
	 * @return the Class
	 * @throws ClassNotFoundException
	 *             If the class cannot be found or read
	 */
	public static Class<?> classForName(String className) throws ClassNotFoundException {
		return classLoaderWrapper.classForName(className);
	}

	/**
	 * get the Charset
	 * 
	 * @return the Charset
	 */
	public static Charset getCharset() {
		return charset;
	}

	/**
	 * set the Charset
	 * 
	 * @param charset
	 *            Charset
	 */
	public static void setCharset(Charset charset) {
		Resources.charset = charset;
	}

}
