package cn.sylinx.hbatis.ds;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import cn.sylinx.hbatis.log.GLog;

/**
 * 数据源管理单例
 * 
 * @author han
 *
 */
public enum JdbcResourceManager {

	me;

	/**
	 * 所有数据源
	 */
	private Map<String, JdbcResourceWrapper> resourceList = new HashMap<String, JdbcResourceWrapper>();

	/**
	 * 默认数据源名称
	 */
	private String defaultJdbcResourceName = null;

	/**
	 * 获取JdbcResourceManager单例
	 * 
	 * @return JdbcResourceManager
	 */
	public static JdbcResourceManager get() {
		return me;
	}

	/**
	 * 添加数据源
	 * 
	 * @param jdbcResource
	 *            JdbcResourceWrapper对象
	 */
	public void addJdbcResource(JdbcResourceWrapper jdbcResource) {
		synchronized (me) {

			if (resourceList.containsKey(jdbcResource.getResourceName())) {
				GLog.info("jdbc resource name {} override", jdbcResource.getResourceName());
			}

			resourceList.put(jdbcResource.getResourceName(), jdbcResource);
			if (jdbcResource.isDefaultResource()) {
				defaultJdbcResourceName = jdbcResource.getResourceName();
			}
		}
	}

	/**
	 * 移除数据源
	 * 
	 * @param jdbResourceName
	 */
	public void removeJdbcResource(String jdbResourceName) {

		synchronized (me) {

			if (!resourceList.containsKey(jdbResourceName)) {
				GLog.info("jdbc resource name {} not exist", jdbResourceName);

			} else {

				JdbcResourceWrapper jrw = resourceList.remove(jdbResourceName);
				jrw.clear();
				jrw = null;
			}

		}
	}

	/**
	 * 获取默认数据源名称
	 * 
	 * @return String
	 */
	public String getDefaultJdbcResourceName() {

		if (defaultJdbcResourceName != null) {
			return defaultJdbcResourceName;
		}

		if (resourceList.isEmpty()) {
			throw new RuntimeException("jdbc resource is empty");
		}

		JdbcResourceWrapper jrw = useDefaultJdbcResourceWrapper();

		if (jrw == null) {
			throw new RuntimeException("not config default jdbc resource");
		}

		return jrw.getResourceName();

	}

	/**
	 * 根据名称获取相应资源
	 * 
	 * @param jdbcResourceName
	 * @return JdbcResource
	 */
	public JdbcResource get(String jdbcResourceName) {

		JdbcResourceWrapper jrw = resourceList.get(jdbcResourceName);
		return jrw != null ? jrw.getJdbcResource() : null;
	}

	/**
	 * 获取默认资源
	 * 
	 * @return JdbcResource
	 */
	public JdbcResource useDefault() {

		if (defaultJdbcResourceName != null) {

			JdbcResourceWrapper jrw = resourceList.get(defaultJdbcResourceName);

			if (jrw != null) {
				return jrw.getJdbcResource();
			}
		}

		JdbcResourceWrapper jrw = useDefaultJdbcResourceWrapper();

		if (jrw != null) {

			return jrw.getJdbcResource();
		}

		throw new RuntimeException("not config default jdbc resource");
	}

	/**
	 * 使用默认数据源
	 * 
	 * @return
	 */
	JdbcResourceWrapper useDefaultJdbcResourceWrapper() {

		Iterator<Entry<String, JdbcResourceWrapper>> it = resourceList.entrySet().iterator();

		while (it.hasNext()) {
			Entry<String, JdbcResourceWrapper> entry = it.next();
			JdbcResourceWrapper jrw = entry.getValue();
			if (jrw.isDefaultResource()) {
				defaultJdbcResourceName = jrw.getResourceName();
				return jrw;
			}
		}

		return null;
	}

	/**
	 * 清除所有资源
	 */
	public void clear() {
		synchronized (me) {
			resourceList.clear();
		}
	}

}
