package cn.sylinx.horm.resource.parse;

import cn.sylinx.horm.dialect.DbType;
import cn.sylinx.horm.exception.HORMException;
import cn.sylinx.horm.resource.ClasspathSqlResource;
import cn.sylinx.horm.resource.ClasspathSqlResourceManager;
import cn.sylinx.horm.resource.liquor.eval.CodeSpec;
import cn.sylinx.horm.resource.liquor.eval.ParamSpec;
import cn.sylinx.horm.resource.liquor.eval.Scripts;
import cn.sylinx.horm.resource.liquor.ext.LiquorContextHolder;
import cn.sylinx.horm.util.Pair;
import cn.sylinx.horm.util.StrKit;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;

class LiquorSqlParser extends SqlParser {

    static final String LIQUOR_CONTEXT_PARAMS = "liquorParams";

    // 数据库类型，用来解析不同数据库类型的sql resource
    private DbType dbtype;
    private String sqlPostfix;

    public LiquorSqlParser(String sqlPostfix, DbType dbtype) {
        this.sqlPostfix = sqlPostfix;
        this.dbtype = dbtype;
    }

    public DbType getDbtype() {
        return dbtype;
    }

    public void setDbtype(DbType dbtype) {
        this.dbtype = dbtype;
    }

    public String getSqlPostfix() {
        return sqlPostfix;
    }

    public void setSqlPostfix(String sqlPostfix) {
        this.sqlPostfix = sqlPostfix;
    }

    public Pair parseSql(ClasspathSqlResource sqlResource, Map<String, Object> params) {
        String statement = getStatement(sqlResource.getSqlpath());
        return parseSqlUseKey(statement, params);
    }

    @Override
    public String parseSqlPart(ClasspathSqlResource sqlResource, Map<String, Object> params) {
        String statement = getStatement(sqlResource.getSqlpath());
        return parseSqlPartUseKey(sqlResource.getSqlpath(), statement, params);
    }

    /**
     * 获取sql文件内容，可能包含#INCLUDE标签
     *
     * @param sqlpath
     * @return
     */
    protected String getStatement(String sqlpath) {
        String truelySqlPath = parseTruelySqlpath(sqlpath);
        String statement = ClasspathSqlResourceManager.getStatement(truelySqlPath);
        if (StrKit.isBlank(statement)) {
            throw new HORMException("SQL语句为空, 资源:" + truelySqlPath);
        }
        return statement;
    }

    protected String getRealSqlPostfix() {
        return "_" + dbtype.getValue() + sqlPostfix;
    }

    protected String parseTruelySqlpath(String sqlpath) {

        if (dbtype == null) {
            return sqlpath;
        }

        // 开启了数据库类型区分
        String dbtypePostfix = getRealSqlPostfix();
        boolean endsWithDbtypePostfix = sqlpath.endsWith(dbtypePostfix);
        if (endsWithDbtypePostfix) {
            return sqlpath;
        }

        int index = sqlpath.lastIndexOf(sqlPostfix);
        if (index < 1) {
            throw new HORMException("invalid sql resource file");
        }

        return sqlpath.substring(0, index) + dbtypePostfix;
    }

    private Pair parseSqlUseKey(String statement, Map<String, Object> params) {

        CodeSpec codeSpec = new CodeSpec(statement)
                .imports(LiquorContextHolder.class)
                .parameters(new ParamSpec(LIQUOR_CONTEXT_PARAMS, Map.class)) //申明参数
                .contextParameter(LIQUOR_CONTEXT_PARAMS)
                .returnType(String.class); //申明返回类型

        Object ret;
        try {
            ret = Scripts.eval(codeSpec, params == null ? new HashMap<>() : params);
        } catch (InvocationTargetException e) {
            throw new HORMException(e);
        }

        if (ret instanceof Pair) {
            return (Pair) ret;
        }

        return PlaceholderParser.INSTANCE_DEFAULT.parseSql(ret.toString(), params);
    }

    public Pair parseSql(String statement, String type, Map<String, Object> params) {
        return parseSqlUseKey(statement, params);
    }

    private String parseSqlPartUseKey(String sqlKey, String statement, Map<String, Object> params) {
        CodeSpec codeSpec = new CodeSpec(statement)
                .imports(LiquorContextHolder.class)
                .parameters(new ParamSpec(LIQUOR_CONTEXT_PARAMS, Map.class)) //申明参数
                .contextParameter(LIQUOR_CONTEXT_PARAMS)
                .returnType(String.class); //申明返回类型

        Object ret;
        try {
            ret = Scripts.eval(codeSpec, params == null ? new HashMap<>() : params);
        } catch (InvocationTargetException e) {
            throw new HORMException(e);
        }

        return ret.toString();
    }

    @Override
    public String parseSqlPart(String statement, String type, Map<String, Object> params) {
        return parseSqlPartUseKey(null, statement, params);
    }

}