/*
 * Copyright 2024 noear.org and authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.sylinx.horm.resource.liquor.eval;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 代码申明
 *
 * @author noear
 * @since 1.2
 */
public class CodeSpec {
    private final String code;
    private final List<String> imports = new ArrayList<>();
    private ParamSpec[] parameters;
    private Class<?> returnType;
    private boolean cached = true;

    // 上下文（定制）
    private String contextParameter;

    public CodeSpec(String code) {
        this.code = code;
    }

    /**
     * 申明缓存的
     */
    public CodeSpec cached(boolean cached) {
        this.cached = cached;
        return this;
    }

    /**
     * 申明导入
     */
    public CodeSpec imports(Class<?>... imports) {
        for (Class<?> imp : imports) {
            this.imports.add(imp.getCanonicalName());
        }
        return this;
    }

    /**
     * 申明导入
     */
    public CodeSpec imports(String... imports) {
        this.imports.addAll(Arrays.asList(imports));
        return this;
    }

    /**
     * 申明参数
     */
    public CodeSpec parameters(ParamSpec... parameters) {
        this.parameters = parameters;
        return this;
    }

    /**
     * 上下文参数
     *
     * @param contextParameter
     * @return
     */
    public CodeSpec contextParameter(String contextParameter) {
        this.contextParameter = contextParameter;
        return this;
    }

    /**
     * 申明返回类型
     */
    public CodeSpec returnType(Class<?> returnType) {
        this.returnType = returnType;
        return this;
    }

    /**
     * 绑定
     *
     * @param context 上下文
     */
    public Object[] bind(Map<String, Object> context) {
        assert context != null;

        parameters = new ParamSpec[context.size()];
        Object[] args = new Object[context.size()];

        int idx = 0;
        for (Map.Entry<String, Object> entry : context.entrySet()) {
            Object value = entry.getValue();
            parameters[idx] = new ParamSpec(entry.getKey(), value == null ? Object.class : trimedClass(value.getClass()));
            args[idx] = value;
            idx++;
        }

        return args;
    }

    private Class<?> trimedClass(Class<?> clazz) {
        Class<?> targetClass = clazz;
        boolean isPublic = Modifier.isPublic(targetClass.getModifiers());
        while (!isPublic) {
            targetClass = targetClass.getSuperclass();
            isPublic = Modifier.isPublic(targetClass.getModifiers());
        }
        return targetClass;
    }

    //////////////////

    /**
     * 是否缓存
     */
    public boolean isCached() {
        return cached;
    }

    /**
     * 获取代码申明
     */
    public String getCode() {
        return code;
    }

    /**
     * 获取导入申明
     */
    public Collection<String> getImports() {
        return imports;
    }

    /**
     * 获取参数申明
     */
    public ParamSpec[] getParameters() {
        return parameters;
    }

    /**
     * 获取上下文参数
     *
     * @return
     */
    public String getContextParameter() {
        return contextParameter;
    }

    /**
     * 获取返回类型申明
     */
    public Class<?> getReturnType() {
        return returnType;
    }

    //////////////////

    private boolean deepEquals(ParamSpec[] a, ParamSpec[] b) {
        if (a == b)
            return true;
        else if (a == null || b == null)
            return false;
        else if (a.length != b.length)
            return false;
        else {
            for (int i = 0; i < a.length; i++) {
                if (!Objects.equals(a[i], b[i])) {
                    return false;
                }
            }

            return true;
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof CodeSpec)) return false;
        CodeSpec codeSpec = (CodeSpec) o;
        return Objects.equals(code, codeSpec.code) && this.deepEquals(parameters, codeSpec.parameters) && Objects.equals(contextParameter, codeSpec.contextParameter);
    }


    @Override
    public int hashCode() {
        return Objects.hash(code, Arrays.hashCode(parameters), contextParameter);
    }
}