package cn.sylinx.horm.resource.parse;

import java.util.HashMap;
import java.util.Map;

import cn.sylinx.horm.config.OrmConfigHolder;
import cn.sylinx.horm.config.ParseSqlType;
import cn.sylinx.horm.dialect.DbType;
import cn.sylinx.horm.exception.HORMException;
import cn.sylinx.horm.resource.ClasspathSqlResource;
import cn.sylinx.horm.resource.ClasspathSqlResourceManager;
import cn.sylinx.horm.resource.StatementHandler;
import cn.sylinx.horm.util.GLog;
import cn.sylinx.horm.util.Pair;
import cn.sylinx.horm.util.StrKit;

class SystemSqlParser extends SqlParser {

    static String KW_INCLUDE_SQL_BEGIN = "#INC[";
    static String KW_INCLUDE_SQL_END = "]";
    static Map<String, String> incSqlMap = new HashMap<>();

    // 数据库类型，用来解析不同数据库类型的sql resource
    private DbType dbtype;
    private String sqlPostfix;

    public SystemSqlParser(String sqlPostfix, DbType dbtype) {
        this.sqlPostfix = sqlPostfix;
        this.dbtype = dbtype;
    }

    public DbType getDbtype() {
        return dbtype;
    }

    public void setDbtype(DbType dbtype) {
        this.dbtype = dbtype;
    }

    public String getSqlPostfix() {
        return sqlPostfix;
    }

    public void setSqlPostfix(String sqlPostfix) {
        this.sqlPostfix = sqlPostfix;
    }

    public Pair parseSql(ClasspathSqlResource sqlResource, Map<String, Object> params) {
        String statement = getRealStatment(sqlResource.getSqlpath());
        return parseSql(statement, params, sqlResource.isFormat(), sqlResource.getStatementHandler());
    }
    
    @Override
    public String parseSqlPart(ClasspathSqlResource sqlResource, Map<String, Object> params) {
        String statement = getRealStatment(sqlResource.getSqlpath());
        return parseSqlPart(statement, ParseSqlType.SYSTEM.getType(), params);
    }

    /**
     * 获取sql文件内容，可能包含#INCLUDE标签
     * 
     * @param sqlpath
     * @return
     */
    protected String getStatement(String sqlpath) {
        String truelySqlPath = parseTruelySqlpath(sqlpath);
        String statement = ClasspathSqlResourceManager.getStatement(truelySqlPath);
        if (StrKit.isBlank(statement)) {
            throw new HORMException("SQL语句为空, 资源:" + truelySqlPath);
        }
        return statement;
    }

    /**
     * 获取真实sql文件内容
     * 
     * @param sqlpath
     * @return
     */
    protected String getRealStatment(String sqlpath) {
        String sqlContent = getStatement(sqlpath);
        int incIndex = sqlContent.indexOf(KW_INCLUDE_SQL_BEGIN);
        if (incIndex < 0) {
            return sqlContent;
        }

        if (OrmConfigHolder.isDebug()) {
            // 调试模式不缓存
            return parseIncludeSqlContent(sqlContent);
        }

        String cacheKey = sqlpath;
        if (dbtype != null) {
            cacheKey = cacheKey + "|" + dbtype.getValue();
        }

        String cachedSqlContent = incSqlMap.get(cacheKey);
        if (cachedSqlContent != null) {
            return cachedSqlContent;
        }

        synchronized (cacheKey.intern()) {
            cachedSqlContent = parseIncludeSqlContent(sqlContent);
            incSqlMap.put(cacheKey, cachedSqlContent);
        }

        return cachedSqlContent;
    }

    /**
     * 解析#INCLUDE标签
     * 
     * @param sqlContent
     * @return
     */
    protected String parseIncludeSqlContent(String sqlContent) {

        int incIndex = sqlContent.indexOf(KW_INCLUDE_SQL_BEGIN);
        if (incIndex < 0) {
            return sqlContent;
        }

        String incIndexLeft = sqlContent.substring(0, incIndex);
        String incIndexRight = sqlContent.substring(incIndex + KW_INCLUDE_SQL_BEGIN.length());
        int incIndex1 = incIndexRight.indexOf(KW_INCLUDE_SQL_END);
        if (incIndex1 < 0) {
            throw new HORMException("INC语法错误");
        }
        // 找到对应包含的sql路径
        String incSqlPath = incIndexRight.substring(0, incIndex1);
        if (StrKit.isBlank(incSqlPath)) {
            throw new HORMException("INC语法错误，丢失SQL资源路径");
        }

        String incIndex1Right = incIndexRight.substring(incIndex1 + KW_INCLUDE_SQL_END.length());

        return incIndexLeft + getRealStatment(incSqlPath.trim()) + parseIncludeSqlContent(incIndex1Right);
    }

    protected String getRealSqlPostfix() {
        return "_" + dbtype.getValue() + sqlPostfix;
    }

    protected String parseTruelySqlpath(String sqlpath) {

        if (dbtype == null) {
            return sqlpath;
        }

        // 开启了数据库类型区分
        String dbtypePostfix = getRealSqlPostfix();
        boolean endsWithDbtypePostfix = sqlpath.endsWith(dbtypePostfix);
        if (endsWithDbtypePostfix) {
            return sqlpath;
        }

        int index = sqlpath.lastIndexOf(sqlPostfix);
        if (index < 1) {
            throw new HORMException("invalid sql resource file");
        }

        return sqlpath.substring(0, index) + dbtypePostfix;
    }

    private Pair parseSql(String statement, Map<String, Object> params, StatementHandler sqlHandler) {
        return parseSql(statement, params, true, sqlHandler);
    }

    public Pair parseSql(String statement, String type, Map<String, Object> params) {
        return parseSql(statement, params, null);
    }

    @Override
    public String parseSqlPart(String statement, String type, Map<String, Object> params) {
        return parseSql(statement, type, params).getFirst();
    }

    private Pair parseSql(String statement, Map<String, Object> params, boolean format, StatementHandler sqlHandler) {
        TokenHandler handler = createWithParameterMap(params);
        GenericTokenParser gt = new GenericTokenParser(handler);
        Pair tp = gt.parse(statement, format);
        if (sqlHandler != null) {
            String statementReplace = sqlHandler.handle(tp.getObject(0, String.class));
            GLog.debug("changed sql: " + statementReplace);
            return Pair.apply(statementReplace, tp.get(1));
        }
        return tp;
    }

    private TokenHandler createWithParameterMap(Map<String, Object> params) {
        TokenHandler handler = TokenHandler.create();
        handler.setParameterMap(params);
        return handler;
    }

}