package cn.sylinx.horm.config;

import java.util.concurrent.atomic.AtomicBoolean;

import cn.sylinx.horm.core.interceptor.SqlClientInterceptor;
import cn.sylinx.horm.core.stat.SqlStatManager;
import cn.sylinx.horm.exception.HORMException;
import cn.sylinx.horm.proxy.command.interceptor.CommandInterceptor;
import cn.sylinx.horm.proxy.mapper.interceptor.MapperInterceptor;
import cn.sylinx.horm.util.GLog;

/**
 * 全局配置
 * 
 * @author johnhan
 *
 */
public enum OrmConfigHolder {

    ;

    private final static AtomicBoolean inited = new AtomicBoolean(false);
    // default config
    private static OrmConfig ormConfig = new OrmConfig();
    private static ServiceEnvironment serviceEnvironment = ServiceEnvironment.NORMAL;
    private static SqlClientInterceptor sqlClientInterceptor = null;
    private static CommandInterceptor commandInterceptor = null;
    private static MapperInterceptor mapperInterceptor = null;

    public static synchronized void init(ServiceEnvironment serviceEnvironment, OrmConfig ormConfig) {

        if (inited.get()) {
            GLog.info("已初始化");
            return;
        }

        try {
            initInner(serviceEnvironment, ormConfig);
        } catch (Exception e) {
            GLog.error("初始化异常", e);
            throw new HORMException("初始化异常", e);
        } finally {
            inited.set(true);
        }
    }

    private static void initInner(ServiceEnvironment serviceEnvironment, OrmConfig ormConfig) {
        OrmConfigHolder.serviceEnvironment = serviceEnvironment;
        OrmConfigHolder.ormConfig = (OrmConfig) ormConfig.clone();
        initConfig();
        GLog.info("配置初始化成功, OrmConifg:{}", OrmConfigHolder.ormConfig);
    }

    private static void initConfig() {

        if (!isDebug() && isCache()) {
            // 支持缓存
            new CacheLoader(ormConfig).initCache();
        }

        if (!isDebug()) {
            // sql加载
            new SqlResourceLoader(ormConfig).load();
        }

        if (isNormalJavaEnv()) {
            // 加载command
            new CommandLoader(ormConfig).load();
            // 加载mapper
            new MapperLoader(ormConfig).load();
            // jdbc事务隔离级别
            new TransactionIsolationLoader(ormConfig).load();
        }

        // 加载模型映射策略
        new MapStrategyLoader(ormConfig).load();

        // 加载模型到缓存
        if (isNormalJavaEnv() && !isDebug() && ormConfig.isCacheModelOnStart()) {
            new ModelCacheLoader(ormConfig).load();
        }

        // 拦截器 --> SqlClient
        sqlClientInterceptor = new SqlClientInterceptorLoader(ormConfig).load();
        // 拦截器 --> Command
        commandInterceptor = new CommandInterceptorLoader(ormConfig).load();
        // 拦截器 --> Mapper
        mapperInterceptor = new MapperInterceptorLoader(ormConfig).load();

        if (isSqlStatOpen()) {
            // sql统计
            SqlStatManager.start();
        }

        // aviator 函数加载
        new AviatorFunctionLoader().load();
    }

    /**
     * 返回克隆对象
     * 
     * @return
     */
    public static OrmConfig getOrmConfig() {
        return (OrmConfig) ormConfig.clone();
    }

    private static boolean isNormalJavaEnv() {
        return ServiceEnvironment.NORMAL == serviceEnvironment;
    }

    public static ServiceEnvironment getServiceEnvironment() {
        return serviceEnvironment;
    }

    public static String getParseSqlType() {
        return ormConfig.getParseSqlType();
    }

    public static boolean isDebug() {
        return ormConfig.isDebug();
    }

    public static boolean isCacheKeyMd5() {
        return ormConfig.isCacheKeyMd5();
    }

    public static boolean isSqlStatOpen() {
        return ormConfig.isSqlStatOpen() && (ormConfig.getSqlExecuteTimeThreshold() > OrmConfig.DEFAULT_STAT_INTERNAL);
    }

    public static boolean isCache() {
        return ormConfig.isCache();
    }

    public static boolean isOptimisticLockEnable() {
        return ormConfig.isOptimisticLockEnable();
    }

    public static boolean isCaseSensitive() {
        return ormConfig.isCaseSensitive();
    }

    public static long getSqlExecuteTimeThreshold() {
        return ormConfig.getSqlExecuteTimeThreshold();
    }

    public static SqlClientInterceptor getSqlClientInterceptor() {
        return sqlClientInterceptor;
    }

    public static boolean hasSqlClientInterceptor() {
        return sqlClientInterceptor != null;
    }

    public static CommandInterceptor getCommandInterceptor() {
        return commandInterceptor;
    }

    public static boolean hasCommandInterceptor() {
        return commandInterceptor != null;
    }

    public static MapperInterceptor getMapperInterceptor() {
        return mapperInterceptor;
    }

    public static boolean hasMapperInterceptor() {
        return mapperInterceptor != null;
    }
    
    public static String getSqlPostfix() {
        return ormConfig.getSqlPostfix();
    }
}