package cn.sylinx.horm.core;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import cn.sylinx.horm.cache.CacheKeyGenerator;
import cn.sylinx.horm.cache.CacheKitManager;
import cn.sylinx.horm.cache.ICacheKit;
import cn.sylinx.horm.cache.IDataLoader;
import cn.sylinx.horm.config.specific.SpecificConfigHolder;
import cn.sylinx.horm.core.common.Callable;
import cn.sylinx.horm.core.common.PreparedCallWrapper;
import cn.sylinx.horm.core.common.PreparedCallable;
import cn.sylinx.horm.core.common.TypedParameter;
import cn.sylinx.horm.core.datasource.ConnectionProvider;
import cn.sylinx.horm.exception.HORMException;
import cn.sylinx.horm.util.DbKit;
import cn.sylinx.horm.util.GLog;

/**
 * sql client
 * 
 * @author johnhan
 *
 */
abstract class AbstractSqlClient implements SqlClient {

    protected ConnectionProvider connectionProvider;

    public ConnectionProvider getConnectionProvider() {
        return connectionProvider;
    }

    public void setConnectionProvider(ConnectionProvider connectionProvider) {
        this.connectionProvider = connectionProvider;
        if (this.connectionProvider != null) {
            DynamicClient.register(this);
        }
    }

    public SqlClient getClient(String dsName) {
        return DynamicClient.get(dsName);
    }

    public String getDataSourceName() {
        return connectionProvider.getDataSourceName();
    }

    /**
     * 添加
     * 
     * @param conn
     * @param updateSql
     * @param params
     * @return
     * @throws SQLException
     */
    public int insert(final String insertSql, final Object... params) throws SQLException {
        return update(insertSql, params);
    }

    @SuppressWarnings("unchecked")
    @Override
    public int[] insertBatch(String insertSqlTemplateInput, List<Object[]> batchParamsInput) throws SQLException {

        if (batchParamsInput == null || batchParamsInput.isEmpty()) {
            throw new HORMException("批量插入参数丢失");
        }

        return call((d, c, s, p) -> {

            PreparedStatement pst = null;

            try {

                List<Object[]> batchParamsInner = (List<Object[]>) p[0];
                pst = c.prepareStatement(s);
                // 设置参数
                for (Object[] row : batchParamsInner) {
                    setParameters(pst, row);
                    pst.addBatch();
                }
                return pst.executeBatch();

            } finally {
                DbKit.closeQuietly(pst);
            }
        }, insertSqlTemplateInput, new Object[] { batchParamsInput });
    }

    /**
     * 删除
     * 
     * @param conn
     * @param deleteSql
     * @param params
     * @return
     * @throws SQLException
     */
    public int delete(final String deleteSql, final Object... params) throws SQLException {
        return update(deleteSql, params);
    }

    /**
     * 更新
     * 
     * @param conn
     * @param updateSql
     * @param params
     * @return
     * @throws SQLException
     */
    public int update(final String updateSqlInput, final Object... paramsInput) throws SQLException {
        return call((d, c, s, p) -> {
            PreparedStatement pst = null;
            try {
                pst = c.prepareStatement(s);
                // 设置参数
                setParameters(pst, p);
                return pst.executeUpdate();
            } finally {
                DbKit.closeQuietly(pst);
            }
        }, updateSqlInput, paramsInput);
    }

    /**
     * 添加，返回生成的键
     * 
     * @param updateSql
     * @param params
     * @return
     * @throws SQLException
     */
    public Object insertForRetrieval(final String insertSqlInput, final Object... paramsInput) throws SQLException {
        return call((d, c, s, p) -> {
            ResultSet rs = null;
            PreparedStatement pst = null;
            try {
                pst = c.prepareStatement(s, PreparedStatement.RETURN_GENERATED_KEYS);
                // 设置参数
                setParameters(pst, p);
                int result = pst.executeUpdate();
                if (result < 1) {
                    GLog.debug("0 rows inserted, sql:{}, params:{}", s, p);
                }
                rs = pst.getGeneratedKeys();
                Object pk = rs.next() ? rs.getObject(1) : null;
                return pk;
            } finally {
                DbKit.closeQuietly(rs, pst);
            }
        }, insertSqlInput, paramsInput);
    }

    /**
     * Executes the SQL statement in this PreparedStatement object, which may be any
     * kind of SQL statement. Some prepared statements return multiple results; the
     * execute method handles these complex statements as well as the simpler form
     * of statements handled by the methods executeQuery and executeUpdate.
     * 
     * The execute method returns a boolean to indicate the form of the first
     * result. You must call either the method getResultSet or getUpdateCount to
     * retrieve the result; you must call getMoreResults to move to any subsequent
     * result(s).
     * 
     * @param exeSql
     * @param params
     * @return true if the first result is a ResultSet object; false if the first
     *         result is an update count or there is no result
     * @throws SQLException
     */
    public boolean execute(final String exeSqlInput, final Object... paramsInput) throws SQLException {
        return call((d, c, s, p) -> {
            PreparedStatement pst = null;
            try {
                pst = c.prepareStatement(s);
                if (p != null) {
                    setParameters(pst, p);
                }
                return pst.execute();
            } finally {
                DbKit.closeQuietly(pst);
            }
        }, exeSqlInput, paramsInput);
    }

    /**
     * 查询
     * 
     * @param conn
     * @param sql
     * @param params
     * @return
     * @throws SQLException
     */
    public List<Object[]> query(final String querySql, final Object... params) throws SQLException {

        String dsName = getDataSourceName();

        if (!SpecificConfigHolder.isCache(dsName)) {
            // 不支持缓存
            return queryFromDB(querySql, params);
        }

        ICacheKit cacheKit = CacheKitManager.get();
        if (cacheKit == null) {
            // 未设置缓存
            return queryFromDB(querySql, params);
        }
        Object key = CacheKeyGenerator.generateCacheKey(getDataSourceName(), "SqlClient.query", querySql, params);
        String cacheKey = String.valueOf(key);
        GLog.debug(
                "use cache query: 'public List<Object[]> query(final String querySql, final Object... params)', cache key:{}",
                cacheKey);
        return cacheKit.get(cacheKey, new IDataLoader() {
            @Override
            public List<Object[]> load() {
                try {
                    return queryFromDB(querySql, params);
                } catch (SQLException e) {
                    GLog.error("queryFromDB error", e);
                    throw new HORMException(e);
                }
            }
        });
    }

    private List<Object[]> queryFromDB(final String querySqlInput, final Object... paramsInput) throws SQLException {
        return call((d, c, s, p) -> {
            ResultSet rs = null;
            PreparedStatement pst = null;
            try {
                List<Object[]> result = new ArrayList<>();
                pst = c.prepareStatement(s);
                // 设置参数
                setParameters(pst, p);
                rs = pst.executeQuery();
                ResultSetMetaData rsmetas = rs.getMetaData();
                int columnCount = rsmetas.getColumnCount();
                while (rs.next()) {
                    Object[] row = new Object[columnCount];
                    for (int i = 1, len = columnCount + 1; i < len; ++i) {
                        row[i - 1] = rs.getObject(i);
                    }
                    result.add(row);
                }
                return result;
            } finally {
                DbKit.closeQuietly(pst);
            }
        }, querySqlInput, paramsInput);
    }

    /**
     * 查询1列
     * 
     * @param querySql 查询sql
     * @param params   查询参数
     * @return
     * @throws SQLException
     */
    public List<Object> queryForSingleColumn(final String querySql, final Object... params) throws SQLException {

        String dsName = getDataSourceName();

        if (!SpecificConfigHolder.isCache(dsName)) {
            // 不支持缓存
            return queryForSingleColumnFromDB(querySql, params);
        }

        ICacheKit cacheKit = CacheKitManager.get();
        if (cacheKit == null) {
            // 未设置缓存
            return queryForSingleColumnFromDB(querySql, params);
        }
        Object key = CacheKeyGenerator.generateCacheKey(getDataSourceName(), "SqlClient.queryForSingleColumn", querySql,
                params);
        String cacheKey = String.valueOf(key);
        GLog.debug(
                "use cache query: 'public List<Object> queryForSingleColumn(final String querySql, final Object... params)', cache key:{}",
                cacheKey);
        return cacheKit.get(cacheKey, new IDataLoader() {
            @Override
            public List<Object> load() {
                try {
                    return queryForSingleColumnFromDB(querySql, params);
                } catch (SQLException e) {
                    GLog.error("queryForSingleColumnFromDB error", e);
                    throw new HORMException(e);
                }
            }
        });
    }

    private List<Object> queryForSingleColumnFromDB(final String querySqlInput, final Object... paramsInput)
            throws SQLException {
        return call((d, c, s, p) -> {
            ResultSet rs = null;
            PreparedStatement pst = null;
            try {
                List<Object> result = new ArrayList<>();
                pst = c.prepareStatement(s);
                // 设置参数
                setParameters(pst, p);
                rs = pst.executeQuery();
                while (rs.next()) {
                    result.add(rs.getObject(1));
                }
                return result;
            } finally {
                DbKit.closeQuietly(pst);
            }
        }, querySqlInput, paramsInput);
    }

    /**
     * 自定义操作
     * 
     * @param <T>
     * @param callable
     * @return
     * @throws SQLException
     */
    public <T> T call(Callable<T> callable) throws SQLException {
        Connection conn = connectionProvider.getConnection();
        try {
            return callable.call(conn);
        } finally {
            connectionProvider.releaseConnection(conn);
        }
    }

    protected <T> T call(PreparedCallable<T> preparedCallable, String sql, Object[] params) throws SQLException {

        String datasourceName = connectionProvider.getDataSourceName();
        Connection conn = connectionProvider.getConnection();
        PreparedCallWrapper callWrapper = null;
        PreparedCallable<T> callableDelegate = null;

        try {
            callWrapper = PreparedCallWrapperManager.getCallWrapper();
            callableDelegate = callWrapper == null ? preparedCallable : callWrapper.wrap(preparedCallable);
            return callableDelegate.call(datasourceName, conn, sql, params);
        } finally {
            connectionProvider.releaseConnection(conn);

            // earliy gc
            callableDelegate = null;
            callWrapper = null;
            conn = null;
            datasourceName = null;
        }
    }

    protected void setParameters(final PreparedStatement pst, final Object... params) throws SQLException {
        if (params != null) {
            for (int i = 0, size = params.length; i < size; ++i) {

                if (params[i] instanceof TypedParameter) {
                    pst.setObject(i + 1, ((TypedParameter) params[i]).getParameter());
                } else {
                    pst.setObject(i + 1, params[i]);
                }
            }
        }
    }
}