package cn.sylinx.horm.starter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import cn.sylinx.horm.cache.impl.GuavaCacheConfig;
import cn.sylinx.horm.config.OrmConfig;
import cn.sylinx.horm.config.OrmConfigHolder;
import cn.sylinx.horm.config.ServiceEnvironment;
import cn.sylinx.horm.core.DynamicClient;
import cn.sylinx.horm.core.OrmClient;
import cn.sylinx.horm.core.TransactionSupportOrmClient;
import cn.sylinx.horm.core.datasource.NamedDataSource;
import cn.sylinx.horm.dialect.DbType;
import cn.sylinx.horm.dialect.DialectFactory;
import cn.sylinx.horm.exception.HORMException;
import cn.sylinx.horm.pool.DataSourceWrapperFactory;
import cn.sylinx.horm.resource.io.Resources;
import cn.sylinx.horm.resource.parse.SqlParser;
import cn.sylinx.horm.transaction.jdbc.JdbcTransactionalConnectionProvider;
import cn.sylinx.horm.util.ExceptionCatcher;
import cn.sylinx.horm.util.GLog;
import cn.sylinx.horm.util.RelaxedPropertyResolver;
import cn.sylinx.horm.util.StrKit;
import cn.sylinx.horm.util.Tuple;

/**
 * 通用java项目启动器
 * 
 * @author johnhan
 *
 */
public class CommonStarter {

	public final static String CONFIG_FILE = "h-orm.properties";
	private static final String MULTI_DATASOURCE_PREFIX = "horm.config.datasource.multids[";
	private static final String MULTI_DATASOURCE_KEY_TEMPLATE = "horm.config.datasource.multids[%d].";

	private String config = CONFIG_FILE;
	private Properties p;

	public CommonStarter() {
		this(null);
	}

	public CommonStarter(String config) {
		if (config != null && !"".equals(config)) {
			this.config = config;
		}
		init();
	}

	private void init() {
		try {
			p = Resources.getResourceAsProperties(config);
		} catch (IOException e) {
			// 配置文件未找到
			GLog.error("初始化异常", e);
			throw new HORMException("初始化异常", e);
		}
	}

	public void start() {
		// 解析配置
		Tuple t = parseProperties();
		OrmConfig ormConfig = t.getObject(0);
		List<NamedDataSource> dataSourceList = t.getObject(1);
		// 初始化SqlClient
		initSqlClient(dataSourceList, ormConfig);
		// 初始化应用配置
		OrmConfigHolder.init(ServiceEnvironment.NORMAL, ormConfig);
	}

	public static class NamedDataSourcePrimary extends NamedDataSource {

		private boolean primary = false;

		public boolean isPrimary() {
			return primary;
		}

		public void setPrimary(boolean primary) {
			this.primary = primary;
		}
	}

	private void initSqlClient(List<NamedDataSource> dataSourceList, OrmConfig ormConfig) {

		if (dataSourceList == null || dataSourceList.isEmpty()) {
			throw new HORMException("数据源信息丢失");
		}
		if (!isMultiple()) {
			// 单数据源
			initOneSqlClient(dataSourceList.get(0), ormConfig);
		} else {
			List<NamedDataSourcePrimary> transferdNdsp = new ArrayList<>();
			int primaryMult = 0;
			NamedDataSourcePrimary primaryed = null;
			// 多数据源
			for (NamedDataSource namedDataSource : dataSourceList) {
				NamedDataSourcePrimary ndsp = (NamedDataSourcePrimary) namedDataSource;
				if (ndsp.isPrimary()) {
					primaryed = ndsp;
					primaryMult++;
				}
				if (primaryMult > 1) {
					throw new HORMException("primary datasource should be only one!");
				}
				transferdNdsp.add(ndsp);
			}
			NamedDataSourcePrimary primaryJumpQueue = null;
			if (primaryed == null) {
				// 都没有设置primary datasource，则取第1个
				primaryed = transferdNdsp.get(0);
			}

			primaryJumpQueue = new NamedDataSourcePrimary();
			primaryJumpQueue.setDataSource(primaryed.getDataSource());
			primaryJumpQueue.setDataSourceName(DynamicClient.DEFAULT_DS_NAME);
			primaryJumpQueue.setDbType(primaryed.getDbType());
			primaryJumpQueue.setPrimary(true);
			
			GLog.info("default SqlClient --> [" + primaryed.getDataSourceName() + "]");
			
			transferdNdsp.add(primaryJumpQueue);
			// 初始化
			transferdNdsp.forEach(ndsp -> initOneSqlClient(ndsp, ormConfig));
		}
	}

	private void initOneSqlClient(NamedDataSource signleNamedDataSource, OrmConfig ormConfig) {
		OrmClient ormClient = new TransactionSupportOrmClient();
		ormClient.setDialect(DialectFactory.createDialect(signleNamedDataSource.getDbType()));
		ormClient.setSqlParser(
				new SqlParser(ormConfig.isParseSqlPathDbtype() ? signleNamedDataSource.getDbType() : null));
		// 此处会将OrmClient句柄保存内存
		ormClient.setConnectionProvider(new JdbcTransactionalConnectionProvider(signleNamedDataSource));
	}

	/**
	 * 
	 * 解析配置，获取应用配置 和 数据源信息
	 * 
	 * @return Tuple[0] : OrmConfig, Tuple[1] : List<NamedDataSource>
	 */
	private Tuple parseProperties() {
		List<NamedDataSource> dataSourceList = new ArrayList<>();
		OrmConfig ormConfig = new OrmConfig();
		// 1）检查数据源类型
		boolean multiple = isMultiple();
		// horm.config.parse-sql-path-dbtype=false
		boolean isParseSqlPathDbtype = isParseSqlPathDbtype();
		ormConfig.setParseSqlPathDbtype(isParseSqlPathDbtype);
		// 2）检查数据源配置是否正确
		if (!multiple) {
			// 1.1）检查单数据源是否配置正确，并构造单数据源
			dataSourceList.add(checkAndGetSingleNamedDataSource());
		} else {
			// 1.2）检查多数据源是否配置正确，并且初始化构造多数据源
			dataSourceList.addAll(checkAndGetMultiNamedDataSource());
		}
		// 3）读取其它配置
		setAppConfigs(ormConfig);
		return Tuple.apply(ormConfig, dataSourceList);
	}

	private void setAppConfigs(OrmConfig ormConfig) {

		boolean cache = ExceptionCatcher.call(() -> Boolean.valueOf(getProperty("horm.config.cache", "false")));
		ormConfig.setCache(cache);
		boolean cacheModelOnStart = ExceptionCatcher
				.call(() -> Boolean.valueOf(getProperty("horm.config.cache-model-on-start", "false")));
		ormConfig.setCacheModelOnStart(cacheModelOnStart);
		String commandScanPackage = getProperty("horm.config.command-scan-package");
		ormConfig.setCommandScanPackage(commandScanPackage);
		boolean debug = ExceptionCatcher.call(() -> Boolean.valueOf(getProperty("horm.config.debug", "false")));
		ormConfig.setDebug(debug);

		if (cache) {
			GuavaCacheConfig guavaCacheConfig = new GuavaCacheConfig();
			// horm.config.guava-cache-config.expire-after-write=10
			String expireAfterWriteStr = getProperty("horm.config.guava-cache-config.expire-after-write",
					String.valueOf(GuavaCacheConfig.DEFAULT_EXPIRE_AFTER_WRITE));
			long expireAfterWrite = ExceptionCatcher.call(() -> Long.valueOf(expireAfterWriteStr));
			guavaCacheConfig.setExpireAfterWrite(expireAfterWrite);

			// horm.config.guava-cache-config.maximum-size=400
			String maximumSizeStr = getProperty("horm.config.guava-cache-config.maximum-size",
					String.valueOf(GuavaCacheConfig.DEFAULT_MAXIMUM_SIZE));
			long maximumSize = ExceptionCatcher.call(() -> Long.valueOf(maximumSizeStr));
			guavaCacheConfig.setMaximumSize(maximumSize);
			ormConfig.setGuavaCacheConfig(guavaCacheConfig);
		}

		String modelMapStrategy = getProperty("horm.config.model-map-strategy");
		ormConfig.setModelMapStrategy(modelMapStrategy);
		String modelScanPackage = getProperty("horm.config.model-scan-package");
		ormConfig.setModelScanPackage(modelScanPackage);
		String mapperScanPackage = getProperty("horm.config.mapper-scan-package");
		ormConfig.setMapperScanPackage(mapperScanPackage);
		String mapperPostfix = getProperty("horm.config.mapper-postfix");
		if (!StrKit.isBlank(mapperPostfix)) {
			ormConfig.setMapperPostfix(mapperPostfix);
		}

		boolean caseSensitive = ExceptionCatcher
				.call(() -> Boolean.valueOf(getProperty("horm.config.case-sensitive", "false")));
		ormConfig.setCaseSensitive(caseSensitive);

		boolean optimisticLockEnable = ExceptionCatcher
				.call(() -> Boolean.valueOf(getProperty("horm.config.optimistic-lock-enable", "false")));
		ormConfig.setOptimisticLockEnable(optimisticLockEnable);
		String sqlPath = getProperty("horm.config.sql-path");
		ormConfig.setSqlPath(sqlPath);

		String sqlPostfix = getProperty("horm.config.sql-postfix");
		if (!StrKit.isBlank(sqlPostfix)) {
			ormConfig.setSqlPostfix(sqlPostfix);
		}

		int transactionIsolation = ExceptionCatcher
				.call(() -> Integer.valueOf(getProperty("horm.config.transaction-isolation", "2")));
		ormConfig.setTransactionIsolation(transactionIsolation);

		// sql 监控 --> horm.config.sql-stat-open 
		boolean sqlStatOpen = ExceptionCatcher
				.call(() -> Boolean.valueOf(getProperty("horm.config.sql-stat-open", "false")));
		ormConfig.setSqlStatOpen(sqlStatOpen);

		// sql 监控 --> horm.config.sql-execute-time-threshold
		String sqlExecuteTimeThresholdStr = getProperty("horm.config.sql-execute-time-threshold",
				String.valueOf(OrmConfig.DEFAULT_STAT_THRESHOLD));
		long sqlExecuteTimeThreshold = ExceptionCatcher.call(() -> Long.valueOf(sqlExecuteTimeThresholdStr));
		ormConfig.setSqlExecuteTimeThreshold(sqlExecuteTimeThreshold);
	}

	private List<NamedDataSource> checkAndGetMultiNamedDataSource() {
		Set<String> nameUnique = new HashSet<>(8);
		List<NamedDataSource> dsList = new ArrayList<>();
		// 获取数据源个数
		int count = getMultiDataSourceCount();
		String keyTemplate = MULTI_DATASOURCE_KEY_TEMPLATE;

		for (int i = 0; i < count; ++i) {
			String keyPrefix = String.format(keyTemplate, i);
			NamedDataSource nds = checkAndGetOneNamedDataSource(keyPrefix);
			if (nameUnique.contains(nds.getDataSourceName())) {
				throw new HORMException("datasource name should be unique");
			}
			nameUnique.add(nds.getDataSourceName());
			// horm.config.datasource.multids[1].primary
			// 多数据源下，决定哪个是默认使用
			String primaryKey = keyPrefix + "primary";
			boolean isPrimary = ExceptionCatcher.call(() -> Boolean.valueOf(getProperty(primaryKey, "false")));
			NamedDataSourcePrimary ndsp = toNamedDataSourcePrimary(nds, isPrimary);
			dsList.add(ndsp);
		}
		return dsList;
	}

	private NamedDataSourcePrimary toNamedDataSourcePrimary(NamedDataSource nds, boolean isPrimary) {
		NamedDataSourcePrimary ndsp = new NamedDataSourcePrimary();
		ndsp.setDataSource(nds.getDataSource());
		ndsp.setDataSourceName(nds.getDataSourceName());
		ndsp.setDbType(nds.getDbType());
		ndsp.setPrimary(isPrimary);
		return ndsp;
	}

	// horm.config.datasource.default-
	private NamedDataSource checkAndGetOneNamedDataSource(String prifix) {

		boolean isMultiple = isMultiple();
		String url = checkAndGetProperty(prifix + "url");
		String driver = checkAndGetProperty(prifix + "driver");
		String dbtype = checkAndGetProperty(prifix + "dbtype");
		DbType dbTypeEnum = DbType.getDbType(dbtype);
		GLog.debug("Use DbType:{}", dbTypeEnum.getValue());

		String pooltype = getProperty(prifix + "pooltype");
		String username = getProperty(prifix + "username");
		String password = getProperty(prifix + "password");
		String name = getProperty(prifix + "name");

		Map<String, Object> dsMap = new HashMap<>();
		dsMap.put("dbtype", dbtype);
		dsMap.put("pooltype", pooltype);
		dsMap.put("name", isMultiple ? name : DynamicClient.DEFAULT_DS_NAME);
		dsMap.put("url", url);
		dsMap.put("driver", driver);
		dsMap.put("username", username);
		dsMap.put("password", password);

		// 默认数据源额外配置
		String extConfigKey = prifix + "pool-config";
		Map<String, Object> extConfig = parseExtConfig(extConfigKey);
		// 获取单（默认）数据源
		NamedDataSource namedDataSource = DataSourceWrapperFactory.buildDataSource(dsMap, extConfig);

//		OrmClient ormClient = new TransactionSupportOrmClient();
//		ormClient.setConnectionProvider(new JdbcTransactionalConnectionProvider(namedDataSource));
//		ormClient.setDialect(DialectFactory.createDialect(dbtype));
//		ormClient.setSqlParser(new SqlParser(isParseSqlPathDbtype() ? dbTypeEnum : null));

		return namedDataSource;
	}

	private boolean isParseSqlPathDbtype() {
		return ExceptionCatcher
				.call(() -> Boolean.valueOf(p.getProperty("horm.config.parse-sql-path-dbtype", "false")));
	}

	private boolean isMultiple() {
		return ExceptionCatcher.call(() -> Boolean.valueOf(p.getProperty("horm.config.datasource.multiple", "false")));
	}

	private int getMultiDataSourceCount() {
		int max = -1;
		Set<Object> keySet = p.keySet();
		for (Object keyObj : keySet) {
			String key = keyObj.toString();
			if (key.startsWith(MULTI_DATASOURCE_PREFIX)) {
				int index = getMultiDataSourceIndex(key);
				if (index > max) {
					max = index;
				}
			}
		}
		if (max < 0) {
			throw new HORMException("invalid multi datasource config");
		}
		return max + 1;
	}

	private static int getMultiDataSourceIndex(String key) {
		String keyReplaced = key.replace(MULTI_DATASOURCE_PREFIX, "");
		int index = keyReplaced.indexOf(']');
		if (index < 1) {
			throw new HORMException("invalid multi datasource config");
		}
		String digitStr = keyReplaced.substring(0, index).trim();
		return Integer.valueOf(digitStr);
	}

	private NamedDataSource checkAndGetSingleNamedDataSource() {
		return checkAndGetOneNamedDataSource("horm.config.datasource.default-");
	}

	/**
	 * 获取对应额外配置
	 * 
	 * @param prefix
	 * @return
	 */
	private Map<String, Object> parseExtConfig(String prefix) {
		return new RelaxedPropertyResolver(p, prefix).toMap();
	}

	private String checkAndGetProperty(String key) {
		String v = getProperty(key);
		if (StrKit.isBlank(v)) {
			throw new HORMException("[" + key + "]未设置值");
		}
		return v;
	}

	private String getProperty(String key, String defaultValue) {
		String v = getProperty(key);
		if (v == null) {
			v = defaultValue;
		}
		return v;
	}

	private String getProperty(String key) {
		return p.getProperty(key);
	}

}
