package cn.sylinx.horm.spring.boot;

import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;

import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;

import cn.sylinx.horm.core.DynamicClient;
import cn.sylinx.horm.core.OrmClient;
import cn.sylinx.horm.core.SqlClient;
import cn.sylinx.horm.core.TransactionSupportOrmClient;
import cn.sylinx.horm.core.datasource.ConnectionProvider;
import cn.sylinx.horm.core.datasource.NamedDataSource;
import cn.sylinx.horm.dialect.DbType;
import cn.sylinx.horm.dialect.DialectFactory;
import cn.sylinx.horm.exception.HORMException;
import cn.sylinx.horm.resource.parse.SqlParser;
import cn.sylinx.horm.spring.config.DataSourceConfig;
import cn.sylinx.horm.spring.config.SingleDataSourceConfig;
import cn.sylinx.horm.transaction.jdbc.JdbcTransactionalConnectionProvider;
import cn.sylinx.horm.transaction.spring.SpringDataSourceConnectionProvider;
import cn.sylinx.horm.util.GLog;
import cn.sylinx.horm.util.StrKit;

/**
 * 使用spring数据源初始化
 * 
 * @author johnhan
 *
 */
class SqlClientInitBySpring {

	private HORMBootConfig bootConfig;

	private ApplicationContext applicationContext;

	public SqlClient create() {
		return parseSpringDatasource(bootConfig);
	}

	public SqlClientInitBySpring setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.applicationContext = applicationContext;
		return this;
	}

	public SqlClientInitBySpring setBootConfig(HORMBootConfig bootConfig) {
		this.bootConfig = bootConfig;
		return this;
	}

	private SqlClient parseSpringDatasource(HORMBootConfig bootConfig) {
		SqlClient sqlClient = null;
		if (isSingle(bootConfig)) {
			GLog.debug("初始化单数据源 by spring");
			sqlClient = getSingleSqlClient(bootConfig);
		} else {
			// 多数据源初始化
			GLog.debug("初始化多数据源 by spring");
			sqlClient = getMultiSqlClient(bootConfig);
		}
		return sqlClient;
	}

	private SqlClient getSingleSqlClient(HORMBootConfig bootConfig, String dbtype, String qualifier,
			String dataSourceName) {

		DbType dbTypeEnum = DbType.getDbType(dbtype);
		DataSource dataSource = StrKit.isBlank(qualifier) ? applicationContext.getBean(DataSource.class)
				: applicationContext.getBean(qualifier, DataSource.class);
		if (dataSource == null) {
			GLog.error(
					StrKit.isBlank(qualifier) ? "Default DataSource NOT FOUND" : "DataSource NOT FOUND, qualifier:{}",
					qualifier);
			throw new HORMException("DataSource NOT FOUND");
		}

		NamedDataSource namedDataSource = new NamedDataSource();
		namedDataSource.setDataSource(dataSource);
		namedDataSource.setDbType(dbTypeEnum);
		namedDataSource.setDataSourceName(dataSourceName);
		OrmClient sqlClient = getSqlClient(bootConfig);
		sqlClient.setConnectionProvider(getConnectionProvider(bootConfig, namedDataSource));
		sqlClient.setDialect(DialectFactory.createDialect(dbtype));
		sqlClient.setSqlParser(getSqlParser(bootConfig, dbTypeEnum));
		return sqlClient;
	}

	private SqlClient initOneDatasource(HORMBootConfig bootConfig, SingleDataSourceConfig singleDataSourceConfig) {
		return getSingleSqlClient(bootConfig, singleDataSourceConfig.getDbtype(), singleDataSourceConfig.getName(),
				singleDataSourceConfig.getName());
	}

	private boolean isSingle(HORMBootConfig bootConfig) {
		DataSourceConfig dataSourceConfig = bootConfig.getDatasource();
		if (dataSourceConfig == null) {
			throw new HORMException("数据源配置丢失");
		}
		return !dataSourceConfig.isMultiple();
	}

	private ConnectionProvider getConnectionProvider(HORMBootConfig bootConfig, NamedDataSource namedDataSource) {
		return bootConfig.isTransactionNative() ? new JdbcTransactionalConnectionProvider(namedDataSource)
				: new SpringDataSourceConnectionProvider(namedDataSource);
	}

	private OrmClient getSqlClient(HORMBootConfig bootConfig) {
		return bootConfig.isTransactionNative() ? new TransactionSupportOrmClient() : new OrmClient();
	}

	private SqlParser getSqlParser(HORMBootConfig bootConfig, DbType dbTypeEnum) {
		return new SqlParser(bootConfig.isParseSqlPathDbtype() ? dbTypeEnum : null);
	}

	private SqlClient getSingleSqlClient(HORMBootConfig bootConfig) {
		return getSingleSqlClient(bootConfig, bootConfig.getDatasource().getDefaultDbtype(), null,
				DynamicClient.DEFAULT_DS_NAME);
	}

	private SqlClient getMultiSqlClient(HORMBootConfig bootConfig) {
		List<SingleDataSourceConfig> multids = bootConfig.getDatasource().getMultids();
		if (multids == null || multids.isEmpty()) {
			throw new HORMException("多数据源初始化异常");
		}
		List<SqlClient> sqlClientList = new ArrayList<>();
		SqlClient defaultSqlClient = null;
		for (SingleDataSourceConfig c : multids) {
			SqlClient sqlClientOne = initOneDatasource(bootConfig, c);
			sqlClientList.add(sqlClientOne);
			if (c.getPrimary() != null && c.getPrimary()) {
				defaultSqlClient = sqlClientOne;
			}
		}
		if (defaultSqlClient == null) {
			defaultSqlClient = sqlClientList.get(0);
		}
		return defaultSqlClient;
	}
}
