package cn.sylinx.horm.spring.boot;

import javax.sql.DataSource;

import org.springframework.beans.BeansException;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Condition;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.core.type.AnnotatedTypeMetadata;

import cn.sylinx.horm.config.OrmConfigHolder;
import cn.sylinx.horm.config.ServiceEnvironment;
import cn.sylinx.horm.core.SqlClient;
import cn.sylinx.horm.util.GLog;

/**
 * HORM自动配置
 * 
 * @author johnhan
 *
 */
@Configuration
@ConditionalOnClass({ DataSource.class })
@EnableConfigurationProperties(HORMBootConfig.class)
@AutoConfigureAfter(DataSourceAutoConfiguration.class)
public class HORMAutoConfiguration {

	static class SqlClientUseSpringCondition implements Condition {
		@Override
		public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
			return !new SqlClientUseNativeCondition().matches(context, metadata);
		}
	}

	static class SqlClientUseNativeCondition implements Condition {
		@Override
		public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
			String type = context.getEnvironment().getProperty("horm.config.datasource.use-native", "false");
			String type1 = context.getEnvironment().getProperty("horm.config.datasource.useNative", "false");
			return Boolean.valueOf(type) || Boolean.valueOf(type1);
		}
	}

	@Configuration
	@Conditional(SqlClientUseNativeCondition.class)
	@Import({ AutoConfiguredCommandScannerRegistrar.class, AutoConfiguredModelScannerRegistrar.class,
			AutoConfiguredMapperScannerRegistrar.class })
	static class SqlClientUseNativeAutoConfiguration {

		/**
		 * 不使用spring数据源
		 * 
		 * @param bootConfig
		 * @return
		 */
		@Bean
		@ConditionalOnMissingBean
		public SqlClient sqlClientUseNative(HORMBootConfig bootConfig) {
			GLog.debug("SqlClient[UseNative] inited begin..");
			SqlClient sqlClient = new SqlClientInitByNative().setBootConfig(bootConfig).create();
			OrmConfigHolder.init(ServiceEnvironment.SPRINGBOOT, bootConfig);
			GLog.info("SqlClient[UseNative] inited done");
			return sqlClient;
		}
	}

	@Configuration
	@Conditional(SqlClientUseSpringCondition.class)
	@ConditionalOnBean(DataSource.class)
	@Import({ AutoConfiguredCommandScannerRegistrar.class, AutoConfiguredModelScannerRegistrar.class,
			AutoConfiguredMapperScannerRegistrar.class })
	static class SqlClientInSpringAutoConfiguration implements ApplicationContextAware {

		private ApplicationContext applicationContext;

		@Override
		public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
			this.applicationContext = applicationContext;
		}

		@Bean
		@ConditionalOnMissingBean
		public SqlClient sqlClient(HORMBootConfig bootConfig) {
			GLog.debug("SqlClient[UseSpring] inited begin..");
			SqlClient sqlClient = new SqlClientInitBySpring().setApplicationContext(applicationContext)
					.setBootConfig(bootConfig).create();
			OrmConfigHolder.init(ServiceEnvironment.SPRINGBOOT, bootConfig);
			GLog.info("SqlClient[UseSpring] inited done");
			return sqlClient;
		}
	}
}