package cn.sylinx.horm.model.cache;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import cn.sylinx.horm.config.OrmConfigHolder;
import cn.sylinx.horm.model.anno.AttributeColumnMapping;
import cn.sylinx.horm.model.anno.Column;
import cn.sylinx.horm.model.anno.Model;
import cn.sylinx.horm.model.anno.PrimaryKey;
import cn.sylinx.horm.model.anno.Table;
import cn.sylinx.horm.model.anno.Version;
import cn.sylinx.horm.model.strategy.Strategy;
import cn.sylinx.horm.model.strategy.UnderlinedStrategy;
import cn.sylinx.horm.model.util.ModelUtil;
import cn.sylinx.horm.util.ClassUtil;
import cn.sylinx.horm.util.GLog;

public enum ModelCacheManager {

	;

	private static Strategy DEFAULT_STRATEGY = new UnderlinedStrategy();

	private static Map<String, ModelFabric> cached = new ConcurrentHashMap<String, ModelFabric>();

	public static int cachedSize() {
		return cached.size();
	}

	public static ModelFabric getModelFabric(String clzStr) {

		Class<?> clz = null;
		try {
			clz = Class.forName(clzStr, false, ClassUtil.getDefaultClassLoader());
			return getModelFabric(clz);
		} catch (ClassNotFoundException e) {
			GLog.error("getModelFabric error:", e);
			return null;
		}
	}

	public static ModelFabric getModelFabric(Class<?> clz) {

		ModelFabric mf = cached.get(clz.getName());
		if (mf == null) {
			try {
				mf = cachedOneModel(clz);
				GLog.debug("添加model[{}]至缓存", clz.getName());
				cached.put(clz.getName(), mf);
			} catch (Exception e) {
				GLog.error("cachedOneModel error, clz:" + clz.getName(), e);
			}
		}
		return mf;
	}

	public static void init(Set<String> clzSets) {

		if (clzSets == null || clzSets.isEmpty()) {
			return;
		}

		for (String clzStr : clzSets) {
			try {
				Class<?> clz = Class.forName(clzStr, false, ClassUtil.getDefaultClassLoader());
				Model model = clz.getDeclaredAnnotation(Model.class);
				if (model != null) {
					// 初始化时，只将模型加载到缓存
					ModelFabric mf = cachedOneModel(clz);
					GLog.debug("添加model[{}]至缓存", clzStr);
					cached.put(clzStr, mf);
				}
			} catch (Exception e) {
				GLog.error("init error", e);
			}
		}

		GLog.debug("注册@Model对象完毕，总计:{}", cached.size());
	}

	public static void initByClassSets(Set<Class<?>> clzSets) {
		for (Class<?> clz : clzSets) {
			Model model = clz.getDeclaredAnnotation(Model.class);
			if (model != null) {
				// 初始化时，只将模型加载到缓存

				try {
					ModelFabric mf = cachedOneModel(clz);
					GLog.debug("添加model[{}]至缓存", clz.getName());
					cached.put(clz.getName(), mf);
				} catch (Exception e) {
					GLog.error("initByClassSets error", e);
				}

			}
		}
	}

	private static Map<String, Column> getFieldDescMap(List<Field> fields) {
		Map<String, Column> fieldDesc = new HashMap<>();
		for (Field f : fields) {
			Column cd = f.getAnnotation(Column.class);
			if (cd != null) {
				fieldDesc.put(f.getName(), cd);
			}
		}
		return fieldDesc;
	}

	private static Field parseVersionField(List<Field> fields) {
		for (Field f : fields) {
			Version version = f.getAnnotation(Version.class);
			if (version != null && Version.versionClassList.contains(f.getType())) {
				return f;
			}
		}
		return null;
	}

	private static Map<String, Field> getFieldsMap(List<Field> fields) {
		Map<String, Field> fieldMap = new HashMap<>();
		for (Field f : fields) {
			fieldMap.put(f.getName(), f);

			// 是否大小写敏感
			if (!OrmConfigHolder.isCaseSensitive()) {
				fieldMap.put(f.getName().toUpperCase(), f);
			}

		}
		return fieldMap;
	}

	public static void register(String className) {

		try {
			register(Class.forName(className, false, ClassUtil.getDefaultClassLoader()));
		} catch (Exception e) {
			GLog.error("register model cache error", e);
		}
	}

	public static void register(Class<?> clz) {
		try {
			cachedOneModel(clz);
			GLog.debug("register model cache:{}", clz);
		} catch (Exception e) {
			GLog.error("register model cache error", e);
		}
	}

	private static ModelFabric cachedOneModel(Class<?> clz) throws Exception {

		final AttributeColumnMapping mapping = clz.getAnnotation(AttributeColumnMapping.class);
		final Table table = clz.getAnnotation(Table.class);
		PrimaryKey pk = clz.getAnnotation(PrimaryKey.class);

		List<Field> fields = ModelUtil.getObjectAllFields(clz);

		Map<String, Field> fieldMap = getFieldsMap(fields);
		Map<String, Column> fieldDesc = getFieldDescMap(fields);
		Field versionField = parseVersionField(fields);

		ModelFabric mf = new ModelFabric();
		mf.setClz(clz);
		mf.setPrimaryKey(pk);
		mf.setTable(table);

		mf.setFields(fields);
		mf.setFieldMap(fieldMap);
		mf.setFieldDesc(fieldDesc);

		mf.setMapping(mapping);
		// 乐观锁字段
		mf.setVersionField(versionField);

		Map<String, String> attrMapping = new HashMap<String, String>();

		// 判断全局映射策略
		Strategy globalStrategy = ModelConst.ME.getMappingStrategy();
		if (globalStrategy == null) {

			if (mapping != null) {
				Strategy as = mapping.strategy().getDeclaredConstructor().newInstance();
				attrMapping.putAll(as.createAttrMapping(mf));
			} else {
				attrMapping.putAll(DEFAULT_STRATEGY.createAttrMapping(mf));
			}

		} else {
			// 使用了全局映射策略
			attrMapping.putAll(globalStrategy.createAttrMapping(mf));
		}

		mf.setAttrMapping(attrMapping);
		mf.setJdbcMapping(mf.createJdbcMapping());

		return mf;
	}

	public static void clear() {

		Set<Entry<String, ModelFabric>> entries = cached.entrySet();
		for (Entry<String, ModelFabric> item : entries) {
			item.getValue().clear();
		}
		cached.clear();
	}
}
