package cn.sylinx.horm.config;

import java.util.concurrent.atomic.AtomicBoolean;

import cn.sylinx.horm.core.stat.SqlStatManager;
import cn.sylinx.horm.util.GLog;

/**
 * 全局配置
 * 
 * @author johnhan
 *
 */
public enum OrmConfigHolder {

	;

	private final static AtomicBoolean inited = new AtomicBoolean(false);
	// default config
	private static OrmConfig ormConfig = new OrmConfig();
	private static ServiceEnvironment serviceEnvironment = ServiceEnvironment.NORMAL;

	public static synchronized void init(ServiceEnvironment serviceEnvironment, OrmConfig ormConfig) {

		if (inited.get()) {
			GLog.info("已初始化");
			return;
		}

		try {
			initInner(serviceEnvironment, ormConfig);
		} finally {
			inited.set(true);
		}
	}

	private static void initInner(ServiceEnvironment serviceEnvironment, OrmConfig ormConfig) {
		OrmConfigHolder.serviceEnvironment = serviceEnvironment;
		try {
			OrmConfigHolder.ormConfig = (OrmConfig) ormConfig.clone();
		} catch (CloneNotSupportedException e) {
			GLog.error("OrmConfig clone error", e);
			return;
		}
		initConfig();
		GLog.info("配置初始化成功, OrmConifg:{}", OrmConfigHolder.ormConfig);
	}

	private static void initConfig() {

		if (!isDebug() && isCache()) {
			// 支持缓存
			new CacheLoader(ormConfig).initCache();
		}

		if (!isDebug()) {
			// sql加载
			new SqlResourceLoader(ormConfig).load();
		}

		if (isNormalJavaEnv()) {
			// 加载command
			new CommandLoader(ormConfig).load();
			// 加载mapper
			new MapperLoader(ormConfig).load();
			// jdbc事务隔离级别
			new TransactionIsolationLoader(ormConfig).load();
		}

		// 加载模型映射策略
		new MapStrategyLoader(ormConfig).load();

		// 加载模型到缓存
		if (isNormalJavaEnv() && !isDebug() && ormConfig.isCacheModelOnStart()) {
			new ModelCacheLoader(ormConfig).load();
		}

		if (isSqlStatOpen()) {
			// sql统计
			SqlStatManager.start();
		}
	}

	private static boolean isNormalJavaEnv() {
		return ServiceEnvironment.NORMAL == serviceEnvironment;
	}

	public static ServiceEnvironment getServiceEnvironment() {
		return serviceEnvironment;
	}

	public static boolean isDebug() {
		return ormConfig.isDebug();
	}

	public static boolean isSqlStatOpen() {
		return ormConfig.isSqlStatOpen() && (ormConfig.getSqlExecuteTimeThreshold() > OrmConfig.DEFAULT_STAT_INTERNAL);
	}

	public static boolean isCache() {
		return ormConfig.isCache();
	}

	public static boolean isOptimisticLockEnable() {
		return ormConfig.isOptimisticLockEnable();
	}

	public static boolean isCaseSensitive() {
		return ormConfig.isCaseSensitive();
	}

	public static long getSqlExecuteTimeThreshold() {
		return ormConfig.getSqlExecuteTimeThreshold();
	}
}