package cn.sylinx.horm.spring.boot;

import java.util.ArrayList;
import java.util.List;
import javax.sql.DataSource;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import cn.sylinx.horm.config.DataSourceConfig;
import cn.sylinx.horm.config.OrmConfigHolder;
import cn.sylinx.horm.config.ServiceEnvironment;
import cn.sylinx.horm.config.SingleDataSourceConfig;
import cn.sylinx.horm.core.OrmClient;
import cn.sylinx.horm.core.SqlClient;
import cn.sylinx.horm.core.datasource.NamedDataSource;
import cn.sylinx.horm.dialect.DbType;
import cn.sylinx.horm.dialect.DialectFactory;
import cn.sylinx.horm.exception.HORMException;
import cn.sylinx.horm.transaction.spring.SpringDataSourceConnectionProvider;
import cn.sylinx.horm.util.GLog;

/**
 * 目前只支持spring数据源的注入
 * 
 * <pre>
	spring.datasource.url=jdbc:mysql://xxxxxxxxx:3306/test?useUnicode=true&characterEncoding=utf-8&useSSL=false&allowMultiQueries=true&serverTimezone=Asia/Shanghai
	spring.datasource.username=root
	spring.datasource.password=xxxxx
	
	horm.config.debug=true
	horm.config.cache=true
	horm.config.guava-cache-config.expire-after-write=10
	horm.config.guava-cache-config.maximum-size=400
	horm.config.sql-path=sql
	horm.config.dbtype=mysql
	
	horm.config.datasource.multiple=true
	## 这里的名称为SpringBean中初始化时的标识符
	horm.config.datasource.multids[0].name=datasource1
	horm.config.datasource.multids[0].primary=false
	horm.config.datasource.multids[0].dbtype=mysql
	
	horm.config.datasource.multids[1].name=datasource2
	horm.config.datasource.multids[1].primary=true
	horm.config.datasource.multids[1].dbtype=mysql
	
	## 以下是多数据源配置，在Spring中自行配置
	spring.datasource.ds1.url=jdbc:mysql://xxxxxxx:3306/test?useUnicode=true&characterEncoding=utf-8&useSSL=false&allowMultiQueries=true&serverTimezone=Asia/Shanghai
	spring.datasource.ds1.username=root
	spring.datasource.ds1.password=xxx
	
	spring.datasource.ds2.url=jdbc:mysql://xxxxxxxx:3306/test?useUnicode=true&characterEncoding=utf-8&useSSL=false&allowMultiQueries=true&serverTimezone=Asia/Shanghai&allowPublicKeyRetrieval=true
	spring.datasource.ds2.username=root
	spring.datasource.ds2.password=xxxxx
 * </pre>
 * 
 * @author johnhan
 *
 */
@Configuration
@AutoConfigureAfter(DataSourceAutoConfiguration.class)
@Import(AutoConfiguredCommandScannerRegistrar.class)
public class BootAutoConfig implements InitializingBean {

	@Autowired
	private ApplicationContext applicationContext;

	@Autowired
	private BootConfig bootConfig;

	@Bean
	@ConfigurationProperties(prefix = "horm.config")
	public BootConfig bootConfig() {
		return new BootConfig();
	}

	@Qualifier
	@Bean
	public SqlClient sqlClient(BootConfig bootConfig) {

		GLog.debug("SqlClientApi inited begin.. ");
		OrmClient ormClient = new OrmClient();
		
		if (isSingle(bootConfig)) {
			GLog.debug("初始化单数据源");
			String dbtype = bootConfig.getDbtype();
			DataSource dataSource = applicationContext.getBean(DataSource.class);
			NamedDataSource namedDataSource = new NamedDataSource();
			namedDataSource.setDataSource(dataSource);
			namedDataSource.setDbType(DbType.getDbType(dbtype));
			ormClient.setConnectionProvider(new SpringDataSourceConnectionProvider(namedDataSource));
			ormClient.setDialect(DialectFactory.createDialect(dbtype));
		} else {
			// 多数据源初始化
			GLog.debug("初始化多数据源");
			List<SingleDataSourceConfig> multids = bootConfig.getDatasource().getMultids();
			if (multids == null || multids.isEmpty()) {
				throw new HORMException("多数据源初始化异常");
			}
			
			List<OrmClient> ormClientList = new ArrayList<>();
			OrmClient defaultOrmClient = null;
			for (SingleDataSourceConfig c : multids) {
				OrmClient ormClientOne = initOneDatasource(c);
				ormClientList.add(ormClientOne);
				if (c.getPrimary() != null && c.getPrimary()) {
					defaultOrmClient = ormClientOne;
				}
			}
			if (defaultOrmClient == null) {
				defaultOrmClient = ormClientList.get(0);
			}
			ormClient = defaultOrmClient;
		}
		GLog.info("OrmClient inited done");
		return ormClient;
	}

	private OrmClient initOneDatasource(SingleDataSourceConfig singleDataSourceConfig) {
		
		OrmClient ormClient = new OrmClient();
		String dataSourceName = singleDataSourceConfig.getName();
		String dbtype = singleDataSourceConfig.getDbtype();
		DataSource dataSource = applicationContext.getBean(singleDataSourceConfig.getName(), DataSource.class);
		NamedDataSource namedDataSource = new NamedDataSource();
		namedDataSource.setDataSource(dataSource);
		namedDataSource.setDbType(DbType.getDbType(dbtype));
		namedDataSource.setDataSourceName(dataSourceName);
		ormClient.setConnectionProvider(new SpringDataSourceConnectionProvider(namedDataSource));
		ormClient.setDialect(DialectFactory.createDialect(dbtype));
		return ormClient;
	}

	private boolean isSingle(BootConfig bootConfig) {
		DataSourceConfig dataSourceConfig = bootConfig.getDatasource();
		if (dataSourceConfig == null) {
			return true;
		}
		return !dataSourceConfig.isMultiple();
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		OrmConfigHolder.init(ServiceEnvironment.SPRINGBOOT, bootConfig);
	}
}