package cn.sylinx.horm.core;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;

import cn.sylinx.horm.cache.CacheKeyGenerator;
import cn.sylinx.horm.cache.CacheKitManager;
import cn.sylinx.horm.cache.ICacheKit;
import cn.sylinx.horm.cache.IDataLoader;
import cn.sylinx.horm.config.OrmConfigHolder;
import cn.sylinx.horm.core.common.Callable;
import cn.sylinx.horm.core.common.OrmUtil;
import cn.sylinx.horm.core.common.Page;
import cn.sylinx.horm.core.common.Record;
import cn.sylinx.horm.dialect.Dialect;
import cn.sylinx.horm.dialect.fs.FS;
import cn.sylinx.horm.exception.HORMException;
import cn.sylinx.horm.model.base.BaseModel;
import cn.sylinx.horm.model.cache.ModelCacheUtil;
import cn.sylinx.horm.model.cache.ModelFabric;
import cn.sylinx.horm.resource.ClasspathSqlResource;
import cn.sylinx.horm.resource.parse.SqlParser;
import cn.sylinx.horm.util.DbKit;
import cn.sylinx.horm.util.ExceptionCatcher;
import cn.sylinx.horm.util.GLog;
import cn.sylinx.horm.util.Pair;
import cn.sylinx.horm.util.Tuple;

/**
 * orm support
 * 
 * @author johnhan
 *
 */
public class OrmClient extends AbstractSqlClient {
	
	private Dialect dialect;

	public Dialect getDialect() {
		return dialect;
	}

	public void setDialect(Dialect dialect) {
		this.dialect = dialect;
	}

	@Override
	public <T, R> List<R> queryForSingleColumn(FS<T> fluentSql) {
		Tuple kv = getDialect().getSqlBuilder().buildSelectSQL(fluentSql);
		String sql = kv.getObject(0);
		Object[] params = kv.getObject(1);
		Class<R> modelClass = kv.getObject(2);
		return queryListForSingleColumn(sql, params, modelClass);
	}

	@Override
	public <T, R> R queryFirstForSingleColumn(FS<T> fluentSql) {
		List<R> objectList = queryForSingleColumn(fluentSql);
		return objectList != null && !objectList.isEmpty() ? objectList.get(0) : null;
	}

	@Override
	public <T> T queryFirst(FS<T> fluentSql) {
		List<T> dataList = query(fluentSql);
		return dataList != null && !dataList.isEmpty() ? dataList.get(0) : null;
	}

	@Override
	public int count(FS<?> fluentSql) {
		Pair kv = getDialect().getSqlBuilder().buildCountSQL(fluentSql);
		String sql = kv.getKey();
		Object[] params = kv.getValue();
		Record r = queryRecord(sql, params);
		return r == null ? 0 : Integer.valueOf(r.get("totalCount").toString());
	}

	@Override
	public <T> Page<T> queryPage(FS<T> fluentSql, int pageNumber, int pageSize) {
		Tuple t = getDialect().getSqlBuilder().buildSelectSQL(fluentSql);
		String sql = t.getObject(0);
		Object[] params = t.getObject(1);
		Class<T> modelClass = t.getObject(2);
		return queryPage(sql, pageNumber, pageSize, params, modelClass);
	}

	@Override
	public <T> List<T> query(FS<T> fluentSql) {
		Tuple kv = getDialect().getSqlBuilder().buildSelectSQL(fluentSql);
		String sql = kv.getObject(0);
		Object[] params = kv.getObject(1);
		Class<T> modelClass = kv.getObject(2);
		return queryList(sql, params, modelClass);
	}

	@Override
	public int update(FS<?> fluentSql) {
		Pair kv = getDialect().getSqlBuilder().buildUpdateSQL(fluentSql);
		String sql = kv.getKey();
		Object[] params = kv.getValue();
		return ExceptionCatcher.call(() -> update(sql, params));
	}

	@Override
	public int delete(FS<?> fluentSql) {
		Pair kv = getDialect().getSqlBuilder().buildDeleteSQL(fluentSql);
		String sql = kv.getKey();
		Object[] params = kv.getValue();
		return ExceptionCatcher.call(() -> delete(sql, params));
	}

	@Override
	public Serializable insert(FS<?> fluentSql) {
		Pair kv = getDialect().getSqlBuilder().buildInsertSQL(fluentSql);
		String sql = kv.getKey();
		Object[] params = kv.getValue();
		try {
			Object obj = insertForRetrieval(sql, params);
			if (obj == null) {
				return null;
			}
			if (obj instanceof Serializable) {
				return (Serializable) obj;
			}
			return obj.toString();
		} catch (Exception e) {
			GLog.error("OrmClient.save fs error", e);
			throw new HORMException(e);
		}
	}

	public <T> Serializable save(T t) {
		if (t instanceof BaseModel) {
			if (((BaseModel) t).getGmtCreate() == null) {
				((BaseModel) t).setGmtCreate(new Date());
			}
		}
		Pair pair = getDialect().getSqlBuilder().buildInsertSQL(t);
		String sql = pair.getObject(0);
		Object[] params = pair.getObject(1);
		try {
			Object obj = insertForRetrieval(sql, params);
			if (obj == null) {
				return null;
			}
			if (obj instanceof Serializable) {
				return (Serializable) obj;
			}
			return obj.toString();
		} catch (Exception e) {
			GLog.error("OrmClient.save error", e);
			throw new HORMException(e);
		}
	}

	public <T> int delete(T t) {
		Pair pair = getDialect().getSqlBuilder().buildDeleteSQL(t);
		String sql = pair.getObject(0);
		Object[] params = pair.getObject(1);
		try {
			return delete(sql, params);
		} catch (Exception e) {
			GLog.error("OrmClient.delete error", e);
			throw new HORMException(e);
		}
	}

	public <T> int update(T t) {
		if (t instanceof BaseModel) {
			if (((BaseModel) t).getGmtModify() == null) {
				((BaseModel) t).setGmtModify(new Date());
			}
		}
		Pair pair = getDialect().getSqlBuilder().buildUpdateSQL(t);
		String sql = pair.getObject(0);
		Object[] params = pair.getObject(1);
		try {
			return update(sql, params);
		} catch (Exception e) {
			GLog.error("OrmClient.update error", e);
			throw new HORMException(e);
		}
	}

	public int update(ClasspathSqlResource sqlResource, Map<String, Object> params) {
		Pair tp = SqlParser.parseSql(sqlResource, params);
		String sql = (String) tp.get(0);
		Object[] paramsArray = (Object[]) tp.get(1);
		return ExceptionCatcher.call(() -> update(sql, paramsArray));
	}

	public boolean execute(ClasspathSqlResource sqlResource, Map<String, Object> params) {
		Pair tp = SqlParser.parseSql(sqlResource, params);
		String sql = (String) tp.get(0);
		Object[] paramsArray = (Object[]) tp.get(1);
		return ExceptionCatcher.call(() -> execute(sql, paramsArray));
	}

	public <T> T get(Serializable id, Class<T> modelClass) {
		Pair pair = getDialect().getSqlBuilder().buildQueryByFieldSQL(Arrays.asList(Pair.apply("id", id)), modelClass);
		String sql = pair.getObject(0);
		Object[] params = pair.getObject(1);
		List<T> dataList = queryList(sql, params, modelClass);
		if (dataList != null && !dataList.isEmpty()) {
			return dataList.get(0);
		}
		return null;
	}

	public Page<Record> queryPage(String sql, int pageNumber, int pageSize, Object[] params) {
		Tuple t = buildPaginatorSql(sql, pageNumber, pageSize, params);
		int totalRow = t.getObject(0);
		if (totalRow == 0) {
			// 空
			Page<Record> emptyPage = new Page<>();
			emptyPage.setPageSize(pageSize);
			return emptyPage;
		}
		int totalPage = t.getObject(1);
		String sqlLimit = t.getObject(2);
		Object[] finalParams = t.getObject(3);
		List<Record> dataList = queryRecords(sqlLimit, finalParams);
		Page<Record> page = new Page<>(dataList, pageNumber, pageSize, totalPage, totalRow);
		return page;
	}

	public Page<Record> queryPage(ClasspathSqlResource sqlResource, int pageNumber, int pageSize,
			Map<String, Object> params) {
		Pair tp = SqlParser.parseSql(sqlResource, params);
		String sql = (String) tp.get(0);
		Object[] paramsArray = (Object[]) tp.get(1);
		return queryPage(sql, pageNumber, pageSize, paramsArray);
	}

	public <T> Page<T> queryPage(String sql, int pageNumber, int pageSize, Object[] params, Class<T> clz) {
		Tuple t = buildPaginatorSql(sql, pageNumber, pageSize, params);
		int totalRow = t.getObject(0);
		if (totalRow == 0) {
			// 空
			Page<T> emptyPage = new Page<>();
			emptyPage.setPageSize(pageSize);
			return emptyPage;
		}
		int totalPage = t.getObject(1);
		String sqlLimit = t.getObject(2);
		Object[] finalParams = t.getObject(3);
		List<T> dataList = queryList(sqlLimit, finalParams, clz);
		return new Page<T>(dataList, pageNumber, pageSize, totalPage, totalRow);
	}

	public <T> Page<T> queryPage(ClasspathSqlResource sqlResource, int pageNumber, int pageSize,
			Map<String, Object> params, Class<T> clz) {
		Pair tp = SqlParser.parseSql(sqlResource, params);
		String sql = (String) tp.get(0);
		Object[] paramsArray = (Object[]) tp.get(1);
		return queryPage(sql, pageNumber, pageSize, paramsArray, clz);
	}

	public <R> Page<R> queryPageForSingleColumn(String sql, int pageNumber, int pageSize, Object[] params,
			Class<R> clz) {
		Tuple t = buildPaginatorSql(sql, pageNumber, pageSize, params);
		int totalRow = t.getObject(0);
		if (totalRow == 0) {
			// 空
			Page<R> emptyPage = new Page<>();
			emptyPage.setPageSize(pageSize);
			return emptyPage;
		}
		int totalPage = t.getObject(1);
		String sqlLimit = t.getObject(2);
		Object[] finalParams = t.getObject(3);
		List<R> dataList = queryListForSingleColumn(sqlLimit, finalParams, clz);
		return new Page<R>(dataList, pageNumber, pageSize, totalPage, totalRow);
	}

	public Record queryRecord(String querySql, Object... params) {
		List<Record> records = queryRecords(querySql, params);
		if (records != null && !records.isEmpty()) {
			return records.get(0);
		}
		return null;
	}

	public List<Record> queryRecords(String querySql, Object... params) {
		if (!OrmConfigHolder.isCache()) {
			// 不支持缓存
			return queryRecordsFromDB(querySql, params);
		}
		ICacheKit cacheKit = CacheKitManager.get();
		if (cacheKit == null) {
			// 未设置缓存
			return queryRecordsFromDB(querySql, params);
		}
		Object key = CacheKeyGenerator.generateCacheKey(getDataSourceName() + "OrmClient.queryRecords", querySql,
				params);
		String cacheKey = String.valueOf(key);
		GLog.debug(
				"use cache query: 'public List<Record> queryRecords(String querySql, Object... params)', cache key:{}",
				cacheKey);
		return cacheKit.get(cacheKey, new IDataLoader() {
			@Override
			public List<Record> load() {
				return queryRecordsFromDB(querySql, params);
			}
		});
	}

	private List<Record> queryRecordsFromDB(String querySql, Object... params) {
		return callSilence(conn -> {
			ResultSet rs = null;
			PreparedStatement pst = null;
			List<Record> result = new ArrayList<Record>();
			try {
				pst = conn.prepareStatement(querySql);
				// 设置参数
				setParameters(pst, params);
				rs = pst.executeQuery();
				ResultSetMetaData rsmetas = rs.getMetaData();
				int columnCount = rsmetas.getColumnCount();
				Record tmp = null;
				while (rs.next()) {
					tmp = new Record();
					for (int i = 1, len = columnCount + 1; i < len; ++i) {
						tmp.put(rsmetas.getColumnLabel(i), i, rs.getObject(i));
					}
					result.add(tmp);
				}
				return result;
			} finally {
				DbKit.closeQuietly(rs, pst);
			}
		});
	}

	public <T> List<T> queryList(Class<T> modelClass) {
		return queryList(getDialect().getSqlBuilder().buildSimpleQuery(modelClass), null, modelClass);
	}

	@Override
	public <T> T queryFirst(final String querySql, Object[] params, Class<T> modelClass) {
		List<T> dataList = queryList(querySql, params, modelClass);
		return dataList != null && !dataList.isEmpty() ? dataList.get(0) : null;
	}

	public <T> List<T> queryList(final String querySql, final Object[] params, Class<T> modelClass) {
		
		if (!OrmConfigHolder.isCache()) {
			// 不支持缓存
			return queryListFromDB(querySql, params, modelClass);
		}
		
		ICacheKit cacheKit = CacheKitManager.get();
		if (cacheKit == null) {
			// 未设置缓存
			return queryListFromDB(querySql, params, modelClass);
		}
		Object key = CacheKeyGenerator.generateCacheKey(getDataSourceName() + "OrmClient.queryList", querySql, params);
		String cacheKey = String.valueOf(key);
		GLog.debug(
				"use cache query: 'public <T> List<T> queryList(final String querySql, final Object[] params, Class<T> modelClass)', cache key:{}",
				cacheKey);
		return cacheKit.get(cacheKey, new IDataLoader() {
			@Override
			public List<T> load() {
				return queryListFromDB(querySql, params, modelClass);
			}
		});
	}

	private <T> List<T> queryListFromDB(final String querySql, final Object[] params, Class<T> modelClass) {
		return callSilence(connnetion -> {
			List<T> result = new ArrayList<>();
			ResultSet rs = null;
			PreparedStatement pst = null;
			ModelFabric mf = ModelCacheUtil.getModelFabric(modelClass);
			Map<String, String> mapping = mf.getJdbcMapping();
			Map<String, Field> fieldMap = mf.getFieldMap();
			try {
				pst = connnetion.prepareStatement(querySql);
				// 设置参数
				setParameters(pst, params);
				rs = pst.executeQuery();
				ResultSetMetaData rsmetas = rs.getMetaData();
				int columnCount = rsmetas.getColumnCount();
				String[] labelNames = new String[columnCount + 1];
				for (int i = 1; i < labelNames.length; i++) {
					labelNames[i] = rsmetas.getColumnLabel(i);
				}
				while (rs.next()) {
					T instance = modelClass.newInstance();
					for (int i = 1, len = columnCount + 1; i < len; ++i) {
						Field f = fieldMap.get(mapping.get(labelNames[i]));
						if (f != null) {
							Object tmp1 = OrmUtil.getResult(rs, i, f.getType());
							if (tmp1 != null) {
								f.setAccessible(true);
								f.set(instance, tmp1);
							}
						}
					}
					result.add(instance);
				}
				return result;
			} catch (Exception e) {
				throw new HORMException(e);
			} finally {
				DbKit.closeQuietly(rs, pst);
			}
		});
	}

	public <T> T queryFirstForSingleColumn(ClasspathSqlResource sqlResource, Map<String, Object> params,
			Class<T> modelClass) {
		List<T> dataList = queryListForSingleColumn(sqlResource, params, modelClass);
		if (dataList != null && !dataList.isEmpty()) {
			return dataList.get(0);
		}
		return null;
	}

	public <T> List<T> queryListForSingleColumn(ClasspathSqlResource sqlResource, Map<String, Object> params,
			Class<T> modelClass) {
		Pair tp = SqlParser.parseSql(sqlResource, params);
		String sql = (String) tp.get(0);
		Object[] paramsArray = (Object[]) tp.get(1);
		return queryListForSingleColumn(sql, paramsArray, modelClass);
	}

	public <T> T queryFirstForSingleColumn(final String querySql, final Object[] params, Class<T> modelClass) {
		List<T> dataList = queryListForSingleColumn(querySql, params, modelClass);
		if (dataList != null && !dataList.isEmpty()) {
			return dataList.get(0);
		}
		return null;
	}

	/**
	 * 查询1列
	 * 
	 * @param <T>
	 * @param querySql   查询sql
	 * @param params     查询参数
	 * @param modelClass 该列对应的类型
	 * @return
	 */
	public <T> List<T> queryListForSingleColumn(final String querySql, final Object[] params, Class<T> modelClass) {
		if (!OrmConfigHolder.isCache()) {
			// 不支持缓存
			return queryListForSingleColumnFromDB(querySql, params, modelClass);
		}
		ICacheKit cacheKit = CacheKitManager.get();
		if (cacheKit == null) {
			// 未设置缓存
			return queryListForSingleColumnFromDB(querySql, params, modelClass);
		}
		Object key = CacheKeyGenerator.generateCacheKey(getDataSourceName() + "OrmClient.queryListForSingleColumn",
				querySql, params);
		String cacheKey = String.valueOf(key);
		GLog.debug(
				"use cache query: 'public <T> List<T> queryListForSingleColumn(final String querySql, final Object[] params, Class<T> modelClass)', cache key:{}",
				cacheKey);
		return cacheKit.get(cacheKey, new IDataLoader() {
			@Override
			public List<T> load() {
				return queryListForSingleColumnFromDB(querySql, params, modelClass);
			}
		});
	}

	private <T> List<T> queryListForSingleColumnFromDB(final String querySql, final Object[] params,
			Class<T> modelClass) {
		return callSilence(connnetion -> {
			List<T> result = new ArrayList<>();
			ResultSet rs = null;
			PreparedStatement pst = null;
			try {
				pst = connnetion.prepareStatement(querySql);
				// 设置参数
				setParameters(pst, params);
				rs = pst.executeQuery();
				ResultSetMetaData rsmetas = rs.getMetaData();
				int columnCount = rsmetas.getColumnCount();
				String[] labelNames = new String[columnCount + 1];
				for (int i = 1; i < labelNames.length; i++) {
					labelNames[i] = rsmetas.getColumnLabel(i);
				}
				while (rs.next()) {
					result.add(OrmUtil.getResult(rs, 1, modelClass));
				}
				return result;
			} catch (Exception e) {
				throw new HORMException(e);
			} finally {
				DbKit.closeQuietly(rs, pst);
			}
		});
	}

	public <T> T queryFirst(ClasspathSqlResource sqlResource, Map<String, Object> params, Class<T> modelClass) {
		List<T> dataList = queryList(sqlResource, params, modelClass);
		if (dataList != null && !dataList.isEmpty()) {
			return dataList.get(0);
		}
		return null;
	}

	public <T> List<T> queryList(ClasspathSqlResource sqlResource, Map<String, Object> params, Class<T> modelClass) {
		Pair tp = SqlParser.parseSql(sqlResource, params);
		String sql = (String) tp.get(0);
		Object[] paramsArray = (Object[]) tp.get(1);
		return queryList(sql, paramsArray, modelClass);
	}

	public Record queryRecord(ClasspathSqlResource sqlResource, Map<String, Object> params) {
		List<Record> records = queryRecords(sqlResource, params);
		if (records != null && !records.isEmpty()) {
			return records.get(0);
		}
		return null;
	}

	public List<Record> queryRecords(ClasspathSqlResource sqlResource, Map<String, Object> params) {
		Pair tp = SqlParser.parseSql(sqlResource, params);
		String sql = (String) tp.get(0);
		Object[] paramsArray = (Object[]) tp.get(1);
		return queryRecords(sql, paramsArray);
	}

	protected <T> T callSilence(Callable<T> callable) {
		try {
			return super.call(callable);
		} catch (Exception e) {
			GLog.error("OrmClient.callSilence error", e);
			throw new HORMException(e);
		}
	}

	/**
	 * 解析为分页sql
	 * 
	 * @param sql
	 * @param pageNumber
	 * @param pageSize
	 * @param params
	 * @return Tuple 0:总行数，1:总页数，2:sql，3:参数
	 */
	protected Tuple buildPaginatorSql(String sql, int pageNumber, int pageSize, Object[] params) {
		String preSql = sql;
		Tuple t = getDialect().getSqlBuilder().buildPaginatorSql(preSql, pageNumber, pageSize);
		String sqlCount = t.getObject(0);
		Record r = queryRecord(sqlCount, params);
		int totalRow = r == null ? 0 : Integer.valueOf(r.get("totalCount").toString());
		if (totalRow == 0) {
			// 空
			return Tuple.apply(0, null, null);
		}
		int totalPage = (int) (totalRow / pageSize);
		if (totalRow % pageSize != 0) {
			totalPage++;
		}
		String sqlLimit = t.getObject(1);
		Object[] pms = t.getObject(2);
		int paramSize = params == null ? 0 : params.length;
		int pageParamSize = pms == null ? 0 : pms.length;
		int finalParamSize = paramSize + pageParamSize;
		Object[] finalParams = new Object[finalParamSize];
		// 查询参数
		if (paramSize > 0) {
			for (int i = 0; i < paramSize; ++i) {
				finalParams[i] = params[i];
			}
		}
		// 分页参数
		if (pageParamSize > 0) {
			for (int i = 0; i < pageParamSize; ++i) {
				finalParams[i + paramSize] = pms[i];
			}
		}
		return Tuple.apply(totalRow, totalPage, sqlLimit, finalParams);
	}

	protected void setParameters(PreparedStatement pst, Object... params) throws SQLException {
		if (params != null) {
			for (int i = 0, size = params.length; i < size; i++) {
				OrmUtil.setParameter(pst, i + 1, params[i]);
			}
		}
	}
}