package cn.sylinx.horm.cache.impl;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import cn.sylinx.horm.cache.ICacheKit;
import cn.sylinx.horm.cache.IDataLoader;
import cn.sylinx.horm.util.GLog;

public class GuavaCacheKit implements ICacheKit {

	private Cache<String, Optional<Object>> cache;

	public GuavaCacheKit() {
		this(new GuavaCacheConfig());
	}

	public GuavaCacheKit(GuavaCacheConfig guavaCacheConfig) {

		cache = CacheBuilder.newBuilder()
				// 默认5分钟过期
				.expireAfterWrite(Duration.ofMinutes(guavaCacheConfig.getExpireAfterWrite()))
				// 最大2000个缓存键
				.maximumSize(guavaCacheConfig.getMaximumSize()).build();
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T> T get(String cacheKey, IDataLoader dataLoader) {

		// 解决guava返回为空值报错的问题
		// CacheLoader returned null for key xxxxx
		Callable<Optional<Object>> callable = () -> {
			GLog.debug("loaded from db, cache key: {} ", cacheKey);
			return Optional.ofNullable(dataLoader.load());
		};

		try {
			Optional<Object> wrappedObject = cache.get(cacheKey, callable);
			return wrappedObject.isPresent() ? (T) wrappedObject.get() : null;

		} catch (ExecutionException e) {
			GLog.error("get cache error", e);
		}

		return null;
	}

}
