package cn.suredoc.sdk.device.protocol.broadcast;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.BeanUtilsBean;
import org.apache.commons.beanutils.converters.DateConverter;
import org.apache.commons.beanutils.converters.SqlDateConverter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONValidator;
import com.alibaba.fastjson.JSONValidator.Type;

import cn.suredoc.sdk.device.protocol.broadcast.dto.BleRecordDto;
import cn.suredoc.sdk.device.protocol.broadcast.dto.HrRecordDetailDto;
import cn.suredoc.sdk.device.protocol.broadcast.dto.HrRecordDto;
import cn.suredoc.sdk.device.protocol.broadcast.dto.RopeRecordDetailDto;
import cn.suredoc.sdk.device.protocol.broadcast.dto.RopeRecordDto;

public class EWUtils {
	// 基站时间戳格式：2022/03/03 17:27:27.283
	private static final String timestampFormat = "yyyy/MM/dd HH:mm:ss.SSS";
	// 基站时间戳格式：2022/03/03 17:27:27
	private static final String timestampShortFormat = "yyyy/MM/dd HH:mm:ss";

	/**
	 * 解析广播数据
	 * 
	 * @param jsonString
	 * @return
	 */
	public static BleRecordDto parseBleData(String jsonString) {
		if (StringUtils.isEmpty(jsonString)) {
			return null;
		}
		JSONArray jsonArray = null;
		// 上传时间
		Date uploadTime = new Date();
		JSONValidator jsonValidator = JSONValidator.from(jsonString);
		Type type = jsonValidator.getType();
		if (type == null) {
			return null;
		}
		// 基站MAC
		String gatewayMac = null;
		// 基站时间
		Date hubTime = null;
		if (type == Type.Array) {
			jsonArray = JSONArray.parseArray(jsonString);
			JSONObject firstObj = jsonArray.getJSONObject(0);
			jsonArray.remove(0);
			// 原始数据
			if (firstObj.containsKey("id")) {
				gatewayMac = firstObj.getString("id");
				hubTime = parseTimestamp(firstObj.getString("time"), uploadTime);
			}
		} else if (type == Type.Object) {
			JSONObject jsonData = JSONObject.parseObject(jsonString);
			if (!jsonData.containsKey("payload")) {
				return null;
			}
			JSONValidator payloadValidator = JSONValidator.from(jsonData.getString("payload"));
			if (payloadValidator.getType() == null) {
				return null;
			}
			// 毫秒级时间戳
			if (jsonData.containsKey("timemills")) {
				uploadTime = new Date(jsonData.getLong("timemills"));
			}
			if (payloadValidator.getType() == Type.Array) {
				jsonArray = jsonData.getJSONArray("payload");
				JSONObject firstObj = jsonArray.getJSONObject(0);
				jsonArray.remove(0);
				// 原始数据
				if (firstObj.containsKey("id")) {
					gatewayMac = firstObj.getString("id");
					hubTime = parseTimestamp(firstObj.getString("time"), uploadTime);
				}
			} else if (payloadValidator.getType() == Type.Object) {
				JSONObject payloadObj = jsonData.getJSONObject("payload");
				if (payloadObj.containsKey("upload_datas")) {
					jsonArray = payloadObj.getJSONArray("upload_datas");
					// 实时模式-原始数据
					if (payloadObj.containsKey("base_info")) {
						JSONObject baseInfo = payloadObj.getJSONObject("base_info");
						gatewayMac = baseInfo.getString("id");
						hubTime = parseTimestamp(baseInfo.getString("time"), uploadTime);
					}
				}
			} else {
				return null;
			}

		}
		List<JSONObject> jsonObjectList = JSONObject.parseArray(jsonArray.toJSONString(), JSONObject.class);
		if (CollectionUtils.isEmpty(jsonObjectList)) {
			return null;
		}
		final Date uploadTimestamp = uploadTime;
		return parseDeviceJsonData(jsonObjectList, uploadTimestamp, hubTime, gatewayMac);
	}

	/**
	 * 解析设备蓝牙广播数据
	 * 
	 * @param list
	 * @param uploadTime 上传时间
	 * @param time       基站时间
	 * @param gatewayMac 网关MAC
	 * @return
	 */
	private static BleRecordDto parseDeviceJsonData(List<JSONObject> list, Date uploadTime, Date time,
			String gatewayMac) {
		if (CollectionUtils.isEmpty(list)) {
			return null;
		}
		// 手环数据
		List<HrRecordDetailDto> hrList = new ArrayList<>();
		// 心率带数据
		List<HrRecordDetailDto> miopodHrList = new ArrayList<>();
		// 跳绳数据
		List<RopeRecordDetailDto> ropeList = new ArrayList<>();
		for (JSONObject jsonObj : list) {
			// 需要解析的数据
			String adData = jsonObj.getString("adv");
			if (StringUtils.isEmpty(adData)) {
				continue;
			}
			Integer rssi = jsonObj.getInteger("rssi");
			String mac = jsonObj.getString("mac");
			String name = DeviceProtocolDecoder.getName(adData);
			if (StringUtils.isEmpty(name)) {
				continue;
			}
			String scanData = jsonObj.getString("rsp");
			// 手环数据
			if (name.startsWith("Mio")) {
				HrRecordDto mioBandRecord = DeviceProtocolDecoder.decodeMioBandRecord(adData);
				if (mioBandRecord == null && rssi != null) {
					HrRecordDetailDto dto = new HrRecordDetailDto();
					dto.setTime(time);
					dto.setRssi(rssi);
					dto.setMac(mac);
					hrList.add(dto);
				} else {
					// 过滤无效数据
					if (mioBandRecord.getHr() <= 0) {
						continue;
					}
					HrRecordDetailDto dto = copy(mioBandRecord, HrRecordDetailDto.class);
					dto.setTime(time);
					if (rssi != null) {
						dto.setRssi(rssi);
					}
					hrList.add(dto);
				}
			} else if (name.startsWith("JC-")) {
				if (StringUtils.isEmpty(scanData)) {
					continue;
				}
				RopeRecordDto ropeData = DeviceProtocolDecoder.decodeRopeData(scanData);
				if (ropeData == null) {
					continue;
				}
				// 模式，0：倒计时跳，1：倒计数跳，2：自由跳，3：团队测试模式
				// 广播只处理倒计时跳与团队测试模式
				if (ropeData.getMode() != 0 && ropeData.getMode() != 3) {
					continue;
				}
				RopeRecordDetailDto dto = copy(ropeData, RopeRecordDetailDto.class);
				dto.setTime(time);
				if (rssi != null) {
					dto.setRssi(rssi);
				}
				ropeList.add(dto);
			} else if (name.startsWith("mioPOD")) {
				if (StringUtils.isEmpty(scanData)) {
					continue;
				}
				Integer hr = DeviceProtocolDecoder.decodeMioPodHr(scanData);
				if (hr == null || hr.intValue() == 0) {
					if (rssi == null) {
						continue;
					}
					HrRecordDetailDto lessonRealTimeHrDto = new HrRecordDetailDto();
					lessonRealTimeHrDto.setTime(time);
					lessonRealTimeHrDto.setRssi(rssi);
					lessonRealTimeHrDto.setMac(mac);
					miopodHrList.add(lessonRealTimeHrDto);
				} else {
					HrRecordDetailDto lessonRealTimeHrDto = new HrRecordDetailDto();
					// 设置hr
					lessonRealTimeHrDto.setHr(hr);
					lessonRealTimeHrDto.setTime(time);
					lessonRealTimeHrDto.setMac(mac);
					if (rssi != null) {
						lessonRealTimeHrDto.setRssi(rssi);
					}
					miopodHrList.add(lessonRealTimeHrDto);
				}
			}

		}
		if (CollectionUtils.isEmpty(hrList) && CollectionUtils.isEmpty(ropeList)
				&& CollectionUtils.isEmpty(miopodHrList)) {
			return null;
		}
		BleRecordDto result = new BleRecordDto();
		result.setHrList(hrList);
		result.setRopeList(ropeList);
		result.setMiopodHrList(miopodHrList);
		result.setUploadTime(uploadTime);
		result.setGatewayMac(gatewayMac);
		// 不可用于连接
		result.setGatewayConnectable(false);
		return result;
	}

	/**
	 * 解析基站时间戳<br>
	 * 2022/03/03 17:27:27.283000<br>
	 * 2022/03/03 17:27:27.283 <br>
	 * 2022/03/03 17:27:27
	 * 
	 * @param timestamp
	 * @return
	 */
	private static Date parseTimestamp(String timestamp) {
		if (StringUtils.isEmpty(timestamp)) {
			return null;
		}
		try {
			if (timestamp.length() == timestampShortFormat.length()) {
				return DateUtils.parseDate(timestamp, timestampShortFormat);
			}
			return DateUtils.parseDate(timestamp.substring(0, timestampFormat.length()), timestampFormat);
		} catch (ParseException e) {
			return null;
		}
	}

	/**
	 * 解析基站时间戳，基站时间可能未同步，如果基站时间与上传数据的时间相差5分钟以上，时间取上传数据的时间<br>
	 * 
	 * @param timestamp  基站扫描到数据的时间<br>
	 *                   2022/03/03 17:27:27.283
	 * @param uploadTime 数据上传时间
	 * @return
	 */
	private static Date parseTimestamp(String timestamp, Date uploadTime) {
		Date time = parseTimestamp(timestamp);
		// 基站时间可能未同步，如果基站时间与上传数据的时间相差5分钟以上，时间取上传数据的时间
		if (time == null || time.before(DateUtils.addMinutes(uploadTime, -5))) {
			return uploadTime;
		}
		return time;
	}

	/**
	 * 拷贝属性
	 * 
	 * @param src   源
	 * @param clazz 目标类型
	 * @return
	 */
	private static <T> T copy(Object src, Class<T> clazz) {
		T result = null;
		try {
			result = clazz.getDeclaredConstructor().newInstance();
			BeanUtilsBean.getInstance().getConvertUtils().register(new SqlDateConverter(null), Date.class);
			BeanUtilsBean.getInstance().getConvertUtils().register(new DateConverter(null), Date.class);
			BeanUtils.copyProperties(result, src);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return result;
	}

}
