package cn.suredoc.sdk.device.protocol.broadcast;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.BeanUtilsBean;
import org.apache.commons.beanutils.converters.DateConverter;
import org.apache.commons.beanutils.converters.SqlDateConverter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONValidator;
import com.alibaba.fastjson.JSONValidator.Type;

import cn.suredoc.sdk.device.protocol.broadcast.dto.BleRecordDto;
import cn.suredoc.sdk.device.protocol.broadcast.dto.HrRecordDetailDto;
import cn.suredoc.sdk.device.protocol.broadcast.dto.HrRecordDto;
import cn.suredoc.sdk.device.protocol.broadcast.dto.RopeRecordDetailDto;
import cn.suredoc.sdk.device.protocol.broadcast.dto.RopeRecordDto;

public class DeviceProtocolUtils {

	public static final String DEFAULT_MIOBAND_NAME_REGEX = "Mio .*";
	public static final String DEFAULT_ROPE_NAME_REGEX = "JC-.*";
	public static final String DEFAULT_MIOPOD_NAME_REGEX = "mioPOD .*";

	/**
	 * 解析广播数据
	 * 
	 * @param jsonString 广播数据
	 * @return
	 */
	public static BleRecordDto parseBleData(final String jsonString) {
		return parseBleData(jsonString, DEFAULT_MIOBAND_NAME_REGEX, DEFAULT_ROPE_NAME_REGEX, DEFAULT_MIOPOD_NAME_REGEX);
	}

	/**
	 * 解析广播数据
	 * 
	 * @param jsonString      广播数据
	 * @param miobandRegex    手环广播名正则，例：Mio .*
	 * @param ropeNameRegex   跳绳广播名正则，例：JC-.*
	 * @param miopodNameRegex 心率带广播名正则，例：mioPOD .*
	 * @return
	 */
	public static BleRecordDto parseBleData(final String jsonString, final String miobandRegex,
			final String ropeNameRegex, final String miopodNameRegex) {
		if (StringUtils.isEmpty(jsonString)) {
			return null;
		}
		if (StringUtils.isAnyEmpty(miobandRegex, ropeNameRegex, miopodNameRegex)) {
			throw new IllegalArgumentException("regex is required.");
		}
		JSONArray jsonArray = null;
		// 上传时间
		Date uploadTime = new Date();
		JSONObject jsonData = null;
		JSONValidator jsonValidator = JSONValidator.from(jsonString);
		Type type = jsonValidator.getType();
		if (type == null) {
			return null;
		}
		// 上传数据的基站MAC地址
		String gatewayMac = null;
		if (type == Type.Array) {
			jsonArray = JSONArray.parseArray(jsonString);
		} else if (type == Type.Object) {
			jsonData = JSONObject.parseObject(jsonString);
			if (jsonData.containsKey("payload")) {
				// tencent iot
				// 毫秒级时间戳
				if (jsonData.containsKey("timemills")) {
					uploadTime = new Date(jsonData.getLong("timemills"));
				}
				if (jsonData.containsKey("devicename")) {
					gatewayMac = jsonData.getString("devicename");
				}
				JSONValidator payloadValidator = JSONValidator.from(jsonData.getString("payload"));
				if (payloadValidator.getType() == null) {
					return null;
				}
				if (payloadValidator.getType() == Type.Array) {
					jsonArray = jsonData.getJSONArray("payload");
				} else if (payloadValidator.getType() == Type.Object) {
					return EWUtils.parseBleData(jsonString);
				} else {
					return null;
				}
			} else if (jsonData.containsKey("action")
					&& StringUtils.equals("data.scan", jsonData.getString("action"))) {
				// MQTT
				if (!jsonData.containsKey("data")) {
					return null;
				}
				if (jsonData.containsKey("gateway")) {
					gatewayMac = jsonData.getString("gateway").replaceAll(":", "");
				}
				jsonArray = jsonData.getJSONArray("data");
			} else {
				return null;
			}
		}

		List<JSONObject> jsonObjectList = JSONObject.parseArray(jsonArray.toJSONString(), JSONObject.class);
		if (CollectionUtils.isEmpty(jsonObjectList)) {
			return null;
		}
		JSONObject firstObj = jsonObjectList.get(0);
		// easyway gateway
		if (firstObj.containsKey("id") && (jsonData == null || !jsonData.containsKey("action"))) {
			return EWUtils.parseBleData(jsonString);
		}
		final Date uploadTimestamp = uploadTime;
		// 手环数据
		List<HrRecordDetailDto> hrList = new ArrayList<>();
		// 心率带数据
		List<HrRecordDetailDto> miopodHrList = new ArrayList<>();
		// 跳绳数据
		List<RopeRecordDetailDto> ropeList = new ArrayList<>();

		jsonObjectList.stream().collect(Collectors.groupingBy(json -> json.getString("name"))).forEach((k, v) -> {
			BleRecordDto dealDeviceClassData = parseDeviceJsonDataByName(miobandRegex, ropeNameRegex, miopodNameRegex,
					k, v, uploadTimestamp);
			if (dealDeviceClassData != null && !CollectionUtils.isEmpty(dealDeviceClassData.getHrList())) {
				hrList.addAll(dealDeviceClassData.getHrList());
			}
			if (dealDeviceClassData != null && !CollectionUtils.isEmpty(dealDeviceClassData.getMiopodHrList())) {
				miopodHrList.addAll(dealDeviceClassData.getMiopodHrList());
			}
			if (dealDeviceClassData != null && !CollectionUtils.isEmpty(dealDeviceClassData.getRopeList())) {
				ropeList.addAll(dealDeviceClassData.getRopeList());
			}
		});
		if (CollectionUtils.isEmpty(hrList) && CollectionUtils.isEmpty(miopodHrList)
				&& CollectionUtils.isEmpty(ropeList)) {
			return null;
		}
		BleRecordDto result = new BleRecordDto();
		result.setHrList(hrList);
		result.setRopeList(ropeList);
		result.setMiopodHrList(miopodHrList);
		result.setUploadTime(uploadTime);
		result.setGatewayMac(gatewayMac);
		result.setGatewayConnectable(true);
		return result;
	}

	/**
	 * 根据广播名解析设备蓝牙广播数据
	 * 
	 * @param miobandRegex    手环广播名正则
	 * @param ropeNameRegex   跳绳广播名正则
	 * @param miopodNameRegex 心率带广播名正则
	 * @param name            广播名
	 * @param list            广播数据
	 * @param uploadTime      上传时间
	 * @return
	 */
	private static BleRecordDto parseDeviceJsonDataByName(String miobandRegex, String ropeNameRegex,
			String miopodNameRegex, String name, List<JSONObject> list, Date uploadTime) {
		if (StringUtils.isEmpty(name) || CollectionUtils.isEmpty(list)) {
			return null;
		}
		// 手环+心率带数据
		List<HrRecordDetailDto> hrList = new ArrayList<>();
		// 手环数据
		List<HrRecordDetailDto> bandDataList;
		// 跳绳数据
		List<RopeRecordDetailDto> ropeDataList = null;

		// 心率带数据
		List<HrRecordDetailDto> mioPodDataList = new ArrayList<>();
		if (Pattern.matches(miobandRegex, name)) {// Mio手环:Mio Band,Mio xxx
			bandDataList = parseMioBandData(list, uploadTime);
			if (!CollectionUtils.isEmpty(bandDataList)) {
				hrList.addAll(bandDataList);
			}
		} else if (Pattern.matches(ropeNameRegex, name)) {// 跳绳数据
			ropeDataList = parseRopeData(list, uploadTime);
		} else if (Pattern.matches(miopodNameRegex, name)) {// 心率带
			mioPodDataList = parseMiopodData(list, uploadTime);
		}
		if (CollectionUtils.isEmpty(hrList) && CollectionUtils.isEmpty(ropeDataList)
				&& CollectionUtils.isEmpty(mioPodDataList)) {
			return null;
		}
		BleRecordDto data = new BleRecordDto();
		data.setRopeList(ropeDataList);
		data.setHrList(hrList);
		data.setMiopodHrList(mioPodDataList);
		return data;
	}

	/**
	 * 解析基站时间戳<br>
	 * 2022-03-03 17:27:27.283 CST <br>
	 * 2022-03-03 09:13:46.358 GMT
	 * 
	 * @param timestamp
	 * @return
	 */
	private static Date parseTimestamp(String timestamp) {
		if (StringUtils.isEmpty(timestamp)) {
			return null;
		}
		if (timestamp.endsWith("CST")) {
			try {
				return DateUtils.parseDate(timestamp.substring(0, timestamp.length() - 4), "yyyy-MM-dd HH:mm:ss.SSS");
			} catch (ParseException e) {
				return null;
			}
		} else if (timestamp.endsWith("GMT")) {
			try {
				return DateUtils.addHours(
						DateUtils.parseDate(timestamp.substring(0, timestamp.length() - 4), "yyyy-MM-dd HH:mm:ss.SSS"),
						8);
			} catch (ParseException e) {
				return null;
			}
		}
		return null;
	}

	/**
	 * 解析基站时间戳，基站时间可能未同步，如果基站时间与上传数据的时间相差5分钟以上，时间取上传数据的时间<br>
	 * 
	 * @param timestamp  基站扫描到数据的时间<br>
	 *                   2022-03-03 17:27:27.283 CST <br>
	 *                   2022-03-03 09:13:46.358 GMT<br>
	 * @param uploadTime 数据上传时间
	 * @return
	 */
	private static Date parseTimestamp(String timestamp, Date uploadTime) {
		Date time = parseTimestamp(timestamp);
		// 基站时间可能未同步，如果基站时间与上传数据的时间相差5分钟以上，时间取上传数据的时间
		if (time == null || time.before(DateUtils.addMinutes(uploadTime, -5))) {
			return uploadTime;
		}
		return time;
	}

	/**
	 * 解析手环广播数据
	 * 
	 */
	private static List<HrRecordDetailDto> parseMioBandData(List<JSONObject> jsonData, Date uploadTime) {
		if (jsonData == null || jsonData.isEmpty()) {
			return null;
		}
		List<HrRecordDetailDto> hrList = new ArrayList<>();
		for (JSONObject jsonObj : jsonData) {
			// 数据时间
			Date time = parseTimestamp(jsonObj.getString("timestamp"), uploadTime);
			// 需要解析的数据
			String adData = jsonObj.getString("adData");
			if (StringUtils.isEmpty(adData)) {
				continue;
			}
			HrRecordDto mioBandRecord = DeviceProtocolDecoder.decodeMioBandRecord(adData);
			Integer rssi = jsonObj.getInteger("rssi");
			JSONArray bdaddrs = jsonObj.getJSONArray("bdaddrs");
			if (CollectionUtils.isEmpty(bdaddrs)) {
				continue;
			}
			List<JSONObject> jsonObjectList = JSONObject.parseArray(bdaddrs.toJSONString(), JSONObject.class);
			String bdaddr = jsonObjectList.get(0).getString("bdaddr");
			if (StringUtils.isEmpty(bdaddr)) {
				continue;
			}
			String mac = bdaddr.replaceAll(":", "");
			if (mioBandRecord == null && rssi != null) {
				HrRecordDetailDto dto = new HrRecordDetailDto();
				dto.setTime(time);
				dto.setRssi(rssi);
				dto.setMac(mac);
				hrList.add(dto);
			} else {
				// 过滤无效数据
				if (mioBandRecord.getHr() <= 0) {
					continue;
				}
				HrRecordDetailDto dto = copy(mioBandRecord, HrRecordDetailDto.class);
				dto.setTime(time);
				if (rssi != null) {
					dto.setRssi(rssi);
				}
				hrList.add(dto);
			}
		}
		return hrList;
	}

	/**
	 * 解析跳绳广播数据
	 * 
	 */
	private static List<RopeRecordDetailDto> parseRopeData(List<JSONObject> jsonData, Date uploadTime) {
		if (jsonData == null || jsonData.isEmpty()) {
			return null;
		}
		List<RopeRecordDetailDto> ropeList = new ArrayList<>();
		for (JSONObject ropeJson : jsonData) {
			// 数据时间
			Date time = parseTimestamp(ropeJson.getString("timestamp"), uploadTime);
			// 扫描响应数据
			String scanData = ropeJson.getString("scanData");
			// 广播数据
			String adData = ropeJson.getString("adData");
			if (StringUtils.isEmpty(scanData) && StringUtils.isEmpty(adData)) {
				continue;
			}
			RopeRecordDto ropeData = DeviceProtocolDecoder.decodeRopeData(scanData);
			if (ropeData == null) {
				ropeData = DeviceProtocolDecoder.decodeRopeData(adData);
				if (ropeData == null) {
					continue;
				}
			}
			RopeRecordDetailDto dto = copy(ropeData, RopeRecordDetailDto.class);
			dto.setTime(time);
			Integer rssi = ropeJson.getInteger("rssi");
			if (rssi != null) {
				dto.setRssi(rssi);
			}
			ropeList.add(dto);
		}
		return ropeList;
	}

	/**
	 * 解析心率带广播数据
	 */
	private static List<HrRecordDetailDto> parseMiopodData(List<JSONObject> mioPODList, Date uploadTime) {
		if (mioPODList == null || mioPODList.isEmpty()) {
			return null;
		}
		List<HrRecordDetailDto> hrList = new ArrayList<>();
		for (JSONObject mioPODJson : mioPODList) {
			// 基站传来的数据时间
			Date time = parseTimestamp(mioPODJson.getString("timestamp"), uploadTime);
			// 需要解析的数据
			String scanData = mioPODJson.getString("scanData");
			if (StringUtils.isEmpty(scanData)) {
				continue;
			}
			// 获取mac
			JSONArray bdaddrs = mioPODJson.getJSONArray("bdaddrs");
			if (CollectionUtils.isEmpty(bdaddrs)) {
				continue;
			}
			String mac;
			List<JSONObject> jsonObjectList = JSONObject.parseArray(bdaddrs.toJSONString(), JSONObject.class);
			String bdaddr = jsonObjectList.get(0).getString("bdaddr");
			if (StringUtils.isEmpty(bdaddr)) {
				continue;
			}
			mac = bdaddr.replaceAll(":", "");
			if (StringUtils.isEmpty(mac)) {
				continue;
			}
			Integer hr = DeviceProtocolDecoder.decodeMioPodHr(scanData);
			Integer rssi = mioPODJson.getInteger("rssi");
			if (hr == null || hr.intValue() == 0) {
				if (rssi == null) {
					continue;
				}
				HrRecordDetailDto lessonRealTimeHrDto = new HrRecordDetailDto();
				lessonRealTimeHrDto.setTime(time);
				lessonRealTimeHrDto.setRssi(rssi);
				lessonRealTimeHrDto.setMac(mac);
				hrList.add(lessonRealTimeHrDto);
			} else {
				HrRecordDetailDto lessonRealTimeHrDto = new HrRecordDetailDto();
				// 设置hr
				lessonRealTimeHrDto.setHr(hr);
				lessonRealTimeHrDto.setTime(time);
				lessonRealTimeHrDto.setMac(mac);
				if (rssi != null) {
					lessonRealTimeHrDto.setRssi(rssi);
				}
				hrList.add(lessonRealTimeHrDto);
			}
		}
		return hrList;
	}

	/**
	 * 拷贝属性
	 * 
	 * @param src   源
	 * @param clazz 目标类型
	 * @return
	 */
	private static <T> T copy(Object src, Class<T> clazz) {
		T result = null;
		try {
			result = clazz.getDeclaredConstructor().newInstance();
			BeanUtilsBean.getInstance().getConvertUtils().register(new SqlDateConverter(null), Date.class);
			BeanUtilsBean.getInstance().getConvertUtils().register(new DateConverter(null), Date.class);
			BeanUtils.copyProperties(result, src);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return result;
	}

}
