package cn.suredoc.sdk.device.protocol.broadcast;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import cn.suredoc.sdk.device.protocol.broadcast.dto.HrRecordDto;
import cn.suredoc.sdk.device.protocol.broadcast.dto.RopeRecordDto;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.UnpooledByteBufAllocator;

class DeviceProtocolDecoder {
	/**
	 * UTF-8
	 */
	public static final Charset DEFAULT_CHARSET = Charset.forName("UTF-8");

	public static byte[] toByteArray(String hex) {
		try {
			return Hex.decodeHex(hex.toCharArray());
		} catch (DecoderException e) {
			throw new RuntimeException(e);
		}
	}

	public static String toHex(byte[] bytes) {
		return Hex.encodeHexString(bytes).toUpperCase();
	}

	public static ByteBuf getDefaultByteBuf() {
		return UnpooledByteBufAllocator.DEFAULT.buffer();
	}

	/**
	 * 释放缓存
	 * 
	 * @param byteBuf
	 */
	public static void release(ByteBuf byteBuf) {
		if (byteBuf == null) {
			return;
		}
		try {
			byteBuf.release();
		} catch (Exception e) {
		}
	}

	/**
	 * 解析跳绳广播数据
	 * 
	 * @param scanData 数据包
	 * @return
	 */
	public static RopeRecordDto decodeRopeData(String scanData) {
		if (StringUtils.isEmpty(scanData) || scanData.length() < 44) {
			return null;
		}
		ByteBuf cmdByteBuf = null;
		try {
			cmdByteBuf = getDefaultByteBuf();
			cmdByteBuf.writeBytes(toByteArray(scanData));
			// 数据长度
			short dataLength = cmdByteBuf.readUnsignedByte();
			if (dataLength < 22) {
				return null;
			}
			RopeRecordDto result = new RopeRecordDto();
			// 数据类型
			short type = cmdByteBuf.readUnsignedByte();
			// 厂商数据
			if (type != 0xff) {
				return null;
			}
			// 数据包头
			short head = cmdByteBuf.readUnsignedByte();
			if (head != 0xfc) {
				return null;
			}
			// 协议版本
			short protocol = cmdByteBuf.readUnsignedByte();
			// mac
			byte[] macArr = new byte[6];
			cmdByteBuf.readBytes(macArr);
			ArrayUtils.reverse(macArr);
			String mac = toHex(macArr).toUpperCase();
			result.setMac(mac);
			// 模式 0：倒计时跳模式，1：倒计数跳模式，2：自由跳模式，3：团队测试模式
			short mode = cmdByteBuf.readUnsignedByte();
			result.setMode(mode);
			switch (protocol) {
			case 0xa2: {
				return decode0xa2(cmdByteBuf, protocol, mode, result);
			}
			default:
				return decode0xa0a1(cmdByteBuf, protocol, mode, result);
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		} finally {
			release(cmdByteBuf);
		}
	}

	/**
	 * 解析0xa2版本数据包
	 * 
	 * @param cmdByteBuf
	 * @param protocol
	 * @param mode
	 * @param result
	 * @return
	 */
	private static RopeRecordDto decode0xa2(ByteBuf cmdByteBuf, short protocol, int mode, RopeRecordDto result) {
		// 当前时间
		long beginTime = cmdByteBuf.readUnsignedIntLE();
		result.setBeginTime(beginTime);
		// 跳绳总次数
		int nums = cmdByteBuf.readUnsignedShortLE();
		result.setNums(nums);
		// 设备状态 烧码后，未开始跳绳该字段值 为0x01，跳绳中为0x03，跳绳结束为0x05。
		short status = cmdByteBuf.readUnsignedByte();
		// 结束状态
		result.setEnd(status == 0x05);
		// 历史跳绳个数
		List<Integer> historyNums = new ArrayList<>();
		result.setHistoryNums(historyNums);
		// 10字节
		while (cmdByteBuf.readableBytes() > 3) {
			String paddedBS = StringUtils.leftPad(Integer.toBinaryString(cmdByteBuf.readUnsignedByte()), 8, "0");
			historyNums.add(Integer.parseUnsignedInt(paddedBS.substring(0, 4), 2));
			historyNums.add(Integer.parseUnsignedInt(paddedBS.substring(4, 8), 2));
		}
		// 3字节
		// 历史绊绳次数
		List<Integer> historyTripWireNums = new ArrayList<>();
		result.setHistoryTripWireNums(historyTripWireNums);
		while (cmdByteBuf.readableBytes() > 0) {
			String paddedBS = StringUtils.leftPad(Integer.toBinaryString(cmdByteBuf.readUnsignedByte()), 8, "0");
			for (int i = 0; i < 8; i++) {
				historyTripWireNums.add(Integer.parseUnsignedInt(paddedBS.substring(i, i + 1), 2));
			}
		}
		return result;
	}

	/**
	 * 解析0xa0,0xa1版本数据包
	 * 
	 * @param cmdByteBuf
	 * @param protocol
	 * @param mode
	 * @param result
	 */
	private static RopeRecordDto decode0xa0a1(ByteBuf cmdByteBuf, short protocol, int mode, RopeRecordDto result) {
		// 当前设置信息
		int setting = cmdByteBuf.readUnsignedShortLE();
		// 倒计时跳：设置倒计时秒数；
		if (mode == 0) {
			result.setTotalTimeSetting(setting);
		} else if (mode == 1) {
			// 倒计数跳：设置倒计次次数
			result.setNumsSetting(setting);
		}
		// 开始跳绳时间戳 单位：秒；跳绳中为0，跳绳结束不为0，团队测试模式时为当前时间
		long beginTime = cmdByteBuf.readUnsignedIntLE();
		result.setBeginTime(beginTime);

		// 跳绳总时长
		int sumTimeData = cmdByteBuf.readUnsignedShortLE();
		result.setTotalTime(sumTimeData);
		// 跳绳总次数
		int nums = cmdByteBuf.readUnsignedShortLE();
		result.setNums(nums);
		// 跳绳有效时长
		int effectiveTime = cmdByteBuf.readUnsignedShortLE();
		result.setEffectiveTime(effectiveTime);
		// 设备状态 烧码后，未开始跳绳该字段值 为0x01，跳绳中为0x03，跳绳结束为0x05。
		short deviceStatus = cmdByteBuf.readUnsignedByte();
		if (protocol == 0xa0) {
			result.setEnd(beginTime != 0);
		} else {
			if (mode == 0) {
				result.setEnd(result.getTotalTime() == result.getTotalTimeSetting() || deviceStatus == 5);
			} else {
				result.setEnd(deviceStatus == 5);
			}
		}
		if (!cmdByteBuf.isReadable()) {
			return result;
		}
		// 倒计时跳模式
		if (mode == 0 && cmdByteBuf.readableBytes() >= 6) {
			// 前20秒绊绳次数
			short frontTripWire = cmdByteBuf.readUnsignedByte();
			result.setFrontTripWire(frontTripWire);
			// 前20秒跳绳个数
			short frontNums = cmdByteBuf.readUnsignedByte();
			result.setFrontNums(frontNums);
			// 中间20秒绊绳次数
			short middleTripWire = cmdByteBuf.readUnsignedByte();
			result.setMiddleTripWire(middleTripWire);
			// 中间20秒跳绳个数
			short middleNums = cmdByteBuf.readUnsignedByte();
			result.setMiddleNums(middleNums);
			// 最后20秒绊绳次数
			short backTripWire = cmdByteBuf.readUnsignedByte();
			result.setBackTripWire(backTripWire);
			// 最后20秒跳绳个数
			short backNums = cmdByteBuf.readUnsignedByte();
			result.setBackNums(backNums);
		}
		return result;
	}

	/**
	 * 解析心率带广播心率
	 * 
	 * @param data
	 * @return
	 */
	public static Integer decodeMioPodHr(String data) {
		if (StringUtils.isBlank(data)) {
			return null;
		}
		if (data.length() != 12) {
			return null;
		}
		ByteBuf cmdByteBuf = null;
		try {
			cmdByteBuf = getDefaultByteBuf();
			cmdByteBuf.writeBytes(toByteArray(data));
			// 数据长度
			cmdByteBuf.readByte();
			// 数据类型
			cmdByteBuf.readByte();
			cmdByteBuf.readShort();
			// 数据是否有效 01:有效，00：无效
			byte valid = cmdByteBuf.readByte();
			if (0 == valid) {
				return null;
			}
			// 心率
			int hr = cmdByteBuf.readUnsignedByte();
			return hr;
		} catch (Exception e) {
			throw new RuntimeException(e);
		} finally {
			release(cmdByteBuf);
		}
	}

	/**
	 * MioBand手环广播数据解析
	 */
	public static HrRecordDto decodeMioBandRecord(String data) {
		if (StringUtils.isEmpty(data)) {
			return null;
		}
		if (data.length() < 50 || !data.startsWith("020106")) {
			return null;
		}
		ByteBuf cmdByteBuf = null;
		try {
			cmdByteBuf = getDefaultByteBuf();
			cmdByteBuf.writeBytes(toByteArray(data));
			cmdByteBuf.readByte();
			cmdByteBuf.readShort();
			byte dataLength = cmdByteBuf.readByte();

			/**
			 * 旧版数据包： 0201060FFF6000EFB840F44F172600517A1300050300AC00A5 <br>
			 * 旧版数据包格式 这三项数据将存放于广播数据中厂商指定数据中 <br>
			 * 格式为如下，包含两个字节的低字节在前 <br>
			 * 内存示例 数据类型 数据定义 <br>
			 * FF u8 厂商指定数据类型 <br>
			 * 0060 u16 厂商id <br>
			 * EFB840F44F17 mac mac地址 <br>
			 * 0026 u16 卡路里 千卡 <br>
			 * 51 u8 实时心率 <br>
			 * 7A u8 最大心率 <br>
			 * 0013 u16 有效运动时长
			 */
			HrRecordDto result = new HrRecordDto();
			if (15 == dataLength) {
				// dataType
				cmdByteBuf.readByte();
				// manufacturerId
				cmdByteBuf.readShortLE();
				// mac
				byte[] macArr = new byte[6];
				cmdByteBuf.readBytes(macArr);
				String mac = toHex(macArr).toUpperCase();
				result.setMac(mac);
				// 卡路里
				int cal = cmdByteBuf.readUnsignedShortLE();
				result.setCalorie(cal);
				// 实时心率
				short hr = cmdByteBuf.readUnsignedByte();
				result.setHr(hr);
				// 最大心率
				short hrMax = cmdByteBuf.readUnsignedByte();
				result.setHrMax(hrMax);
				// 有效运动时间(秒)
				int seconds = cmdByteBuf.readUnsignedShortLE();
				result.setSeconds(seconds);
				return result;
			} else if (dataLength == 13) {
				/**
				 * 加设备名数据包-有UUID，无厂商ID
				 * 0201060DFFEFB840F44F173A005E77000009094D696F2042616E64030300AC <br>
				 * 02 长度<br>
				 * 01 数据类型：Flags <br>
				 * 06 flag<br>
				 * 
				 * 0D 长度 <br>
				 * FF 数据类型：厂商数据 <br>
				 * EFB840F44F17 6字节 mac地址<br>
				 * 3A00 2字节 卡路里 单位千卡（小端）：58 <br>
				 * 5E 1字节 实时心率：94 <br>
				 * 77 1字节 最大心率：119 <br>
				 * 0000 2字节 有效运动时长（小端）0<br>
				 * 
				 * 设备名称 09 <br>
				 * 长度 09 <br>
				 * 数据类型：设备名 <br>
				 * 4D696F2042616E64 设备名：Mio Band<br>
				 * 
				 * UUID 03 <br>
				 * 长度 03 <br>
				 * 数据类型：UUID <br>
				 * 00AC UUID
				 */
				// dataType
				cmdByteBuf.readByte();
				// mac
				byte[] macArr = new byte[6];
				cmdByteBuf.readBytes(macArr);
				String mac = toHex(macArr).toUpperCase();
				result.setMac(mac);
				// 卡路里
				int cal = cmdByteBuf.readUnsignedShortLE();
				result.setCalorie(cal);
				// 实时心率
				short hr = cmdByteBuf.readUnsignedByte();
				result.setHr(hr);
				// 最大心率
				short hrMax = cmdByteBuf.readUnsignedByte();
				result.setHrMax(hrMax);
				// 有效运动时间(秒)
				int seconds = cmdByteBuf.readUnsignedShortLE();
				result.setSeconds(seconds);
				return result;
			}
			return null;
		} catch (Exception e) {
			throw new RuntimeException(e);
		} finally {
			release(cmdByteBuf);
		}
	}

	/**
	 * 获取广播名
	 * 
	 * @param data
	 * @return
	 */
	public static String getName(String data) {
		if (StringUtils.isEmpty(data)) {
			return null;
		}
		ByteBuf cmdByteBuf = null;
		String name = null;
		try {
			cmdByteBuf = getDefaultByteBuf();
			cmdByteBuf.writeBytes(toByteArray(data));
			while (cmdByteBuf.readableBytes() > 0) {
				short dataLength = cmdByteBuf.readUnsignedByte();
				short dataType = cmdByteBuf.readUnsignedByte();
				byte[] dataBuf = new byte[dataLength - 1];
				cmdByteBuf.readBytes(dataBuf);
				if (dataType == 9) {
					name = new String(dataBuf);
					break;
				}
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		} finally {
			release(cmdByteBuf);
		}
		return name;
	}

}
