package cn.suredoc.sdk.device.protocol.base.utils;

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Consts;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpClientUtils {

	// 设置请求和传输超时时间
	private static RequestConfig defaultRequestConfig = RequestConfig.custom().setSocketTimeout(5000)
			.setConnectTimeout(5000).setConnectionRequestTimeout(5000).build();

	private static final Logger logger = LoggerFactory.getLogger(HttpClientUtils.class);

	private HttpClientUtils() {
	}

	/**
	 * Get请求,不带参数
	 * 
	 * @param url 资源访问路径
	 * @return String 响应内容
	 */
	public static String get(String url) {
		// Get请求
		HttpGet httpGet = new HttpGet(url);
		httpGet.setHeader("Connection", "close");// 处理CLOSE_WAIT
		String body = null;

		try {

			// 设置参数
			httpGet.setURI(new URI(httpGet.getURI().toString()));

			httpGet.setConfig(defaultRequestConfig);

			// 发送请求
			HttpResponse httpResponse = getHttpclient().execute(httpGet);

			// 获取返回数据
			HttpEntity entity = httpResponse.getEntity();

			body = EntityUtils.toString(entity, Consts.UTF_8);
			if (entity != null) {
				EntityUtils.consume(entity);
			}
		} catch (Exception e) {
			logger.warn("url:" + url + " ," + e.getMessage(), e);
		} finally {
			// 关闭连接,释放资源
			httpGet.abort();
			httpGet.releaseConnection();
		}
		return body;
	}

	/**
	 * get请求 带参数
	 * 
	 * @param url    链接
	 * @param params 参数
	 * @return
	 */
	public static String get(String url, Map<String, Object> params) {
		// Get请求
		HttpGet httpGet = null;
		String body = null;
		try {
			if (MapUtils.isNotEmpty(params)) {
				List<NameValuePair> entityParams = new ArrayList<>(params.size());
				params.forEach((key, value) -> {
					entityParams.add(new BasicNameValuePair(key, (String) value));
				});
				String paramsUrl = EntityUtils.toString(new UrlEncodedFormEntity(entityParams, Consts.UTF_8));
				url = url + "?" + paramsUrl;
			}
			httpGet = new HttpGet(url);
			httpGet.setHeader("Connection", "close");// 处理CLOSE_WAIT
			// 设置参数
			httpGet.setURI(new URI(httpGet.getURI().toString()));

			httpGet.setConfig(defaultRequestConfig);

			// 发送请求
			HttpResponse httpResponse = getHttpclient().execute(httpGet);

			// 获取返回数据
			HttpEntity entity = httpResponse.getEntity();

			body = EntityUtils.toString(entity, Consts.UTF_8);
			// System.out.println("返回JSON body：" + body);
			if (entity != null) {
				EntityUtils.consume(entity);
			}
		} catch (Exception e) {
			logger.warn("url:" + url + " ," + e.getMessage(), e);
		} finally {
			// 关闭连接,释放资源
			httpGet.abort();
			httpGet.releaseConnection();
		}
		return body;
	}

	/**
	 * 
	 * Post请求,带参数
	 * 
	 * @param url    资源访问路径
	 * @param params 参数
	 * @return String 响应内容
	 */
	public static String post(String url, String params) {
		return post(url, params, null);
	}

	/**
	 * 
	 * Post请求,带参数
	 * 
	 * @param url    资源访问路径
	 * @param params 参数
	 * @return String 响应内容
	 */
	public static String post(String url, String params, Map<String, String> headers) {
		// Post请求
		HttpPost httpPost = new HttpPost(url);
		httpPost.setConfig(defaultRequestConfig);
		httpPost.setHeader("Connection", "close");// 处理CLOSE_WAIT
		if (headers != null && !headers.isEmpty()) {
			Set<String> keySet = headers.keySet();
			Iterator<String> iterator = keySet.iterator();
			while (iterator.hasNext()) {
				String key = iterator.next();
				String value = headers.get(key);
				if (!StringUtils.isEmpty(value)) {
					httpPost.addHeader(key, value);
				}
			}
		}

		String body = null;

		try {
			if (!StringUtils.isEmpty(params)) {
				StringEntity se = new StringEntity(params, Consts.UTF_8);
				se.setContentType("application/json");
				// 设置参数
				httpPost.setEntity(se);
			}

			// 发送请求
			HttpResponse httpResponse = getHttpclient().execute(httpPost);
			// 获取返回数据
			HttpEntity entity = httpResponse.getEntity();
			if (entity != null) {
				body = EntityUtils.toString(entity, Consts.UTF_8);
				EntityUtils.consume(entity);
			}

		} catch (Exception e) {
			logger.warn("url:" + url + " ," + e.getMessage(), e);

		} finally {
			// 关闭连接,释放资源
			httpPost.abort();
			httpPost.releaseConnection();
		}
		return body;
	}

	/**
	 * 
	 * delete请求
	 * 
	 * @param url 资源访问路径
	 * @return String 响应内容
	 */
	public static String delete(String url) {
		return delete(url, null);
	}

	/**
	 * 
	 * delete请求
	 * 
	 * @param url     资源访问路径
	 * @param headers 请求头
	 * @return String 响应内容
	 */
	public static String delete(String url, Map<String, String> headers) {
		// Post请求
		HttpDelete httpDelete = new HttpDelete(url);
		httpDelete.setConfig(defaultRequestConfig);
		httpDelete.setHeader("Connection", "close");// 处理CLOSE_WAIT
		if (headers != null && !headers.isEmpty()) {
			Set<String> keySet = headers.keySet();
			Iterator<String> iterator = keySet.iterator();
			while (iterator.hasNext()) {
				String key = iterator.next();
				String value = headers.get(key);
				if (!StringUtils.isEmpty(value)) {
					httpDelete.addHeader(key, value);
				}
			}
		}

		String body = null;

		try {
			// 发送请求
			HttpResponse httpResponse = getHttpclient().execute(httpDelete);

			// 获取返回数据
			HttpEntity entity = httpResponse.getEntity();
			if (entity != null) {
				body = EntityUtils.toString(entity, Consts.UTF_8);
				EntityUtils.consume(entity);
			}

		} catch (Exception e) {
			logger.warn("url:" + url + " ," + e.getMessage(), e);
		} finally {
			// 关闭连接,释放资源
			httpDelete.abort();
			httpDelete.releaseConnection();
		}
		return body;
	}

	public static CloseableHttpClient getHttpclient() {
		CloseableHttpClient httpclient = HttpClients.createDefault();
		return httpclient;
	}

}
