package cn.structure.common.entity;

import cn.structure.common.enums.ResultCodeEnum;
import cn.structure.common.exception.CommonException;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.*;

import java.io.Serializable;
import java.util.Date;
import java.util.List;


/**
 * <p>
 * 出参 ： 返回结果封装 - VO
 * 对微服务比较友好
 * </p>
 *
 * @param <T> data部分的类型
 * @author chuck
 * @version 1.0.1
 * @since 2020-12-26
 */
@ApiModel(description = "出参 ： 返回结果封装 - VO")
@Getter
@Setter
@ToString
@Builder
@EqualsAndHashCode
@NoArgsConstructor
@AllArgsConstructor
public class ResultVO<T> implements Serializable {
    /**
     * 系统级别的CODE码
     */
    @ApiModelProperty(value = "系统级别的CODE码", example = "SUCCESS")
    private String code;
    /**
     * 系统级别的消息内容
     */
    @ApiModelProperty(value = "系统级别的消息内容", example = "成功！")
    private String msg;
    /**
     * 业务级别的code码
     */
    @ApiModelProperty(value = "业务级别的code码", example = "SUCCESS")
    private String subCode;
    /**
     * 业务级别的消息内容
     */
    @ApiModelProperty(value = "业务级别的消息内容", example = "成功！")
    private String subMsg;
    /**
     * 系统响应的时间戳
     */
    @ApiModelProperty(value = "系统响应的时间戳", example = "112345644446")
    private Long timestamp;
    /**
     * 系统响应的回参数据是一个泛型
     */
    @ApiModelProperty(value = "系统响应的回参数据是一个泛型", example = "{}")
    private T data;

    /**
     * builder 一个成功的ResultVO
     *
     * @param <T>
     * @return
     */
    public static <T> ResultVO<T> success() {
        return success(null);
    }

    /**
     * builder 一个成功的ResultVO 入参为出参data即可
     *
     * @param d
     * @param <T>
     * @return
     */
    public static <T> ResultVO<T> success(T d) {
        return ResultVO.<T>builder()
                .data(d)
                .code(ResultCodeEnum.SUCCESS.getCode())
                .msg(ResultCodeEnum.SUCCESS.getMsg())
                .subCode(ResultCodeEnum.SUCCESS.getCode())
                .subMsg(ResultCodeEnum.SUCCESS.getMsg())
                .timestamp(System.currentTimeMillis())
                .build();
    }

    /**
     * 异常的处理
     *
     * @param message 异常信息
     * @return
     */
    public static <T> ResultVO<T> exception(String message) {
        return ResultVO.<T>builder()
                .code(ResultCodeEnum.ERR.getCode())
                .msg(message)
                .timestamp(System.currentTimeMillis())
                .build();
    }

    /**
     * 异常的处理
     *
     * @param ce 公共异常类
     * @return
     */
    public static <T> ResultVO<T> exception(CommonException ce) {
        return exception(ce.getMessage(), ce.getCode(), ce.getMsg());
    }

    /**
     * 异常的处理
     *
     * @param msg     异常消息
     * @param subCode 业务code
     * @param subMsg  业务消息
     * @return
     */
    public static <T> ResultVO<T> exception(String msg, String subCode, String subMsg) {
        return ResultVO.<T>builder()
                .code(ResultCodeEnum.ERR.getCode())
                .msg(msg)
                .subCode(subCode)
                .subMsg(subMsg)
                .timestamp(System.currentTimeMillis())
                .build();
    }

    /**
     * 业务失败的构建
     *
     * @param subCode 业务code
     * @param subMsg  业务消息
     * @return
     */
    public static <T> ResultVO<T> fail(String subCode, String subMsg) {
        return fail(subCode, subMsg, null);
    }

    /**
     * 业务失败的构建
     *
     * @param subCode 业务code
     * @param subMsg  业务消息
     * @return
     */
    public static <T> ResultVO<T> fail(Integer subCode, String subMsg) {
        return fail(subCode.toString(), subMsg);
    }

    /**
     * 验证失败但是仍然要返回业务状态吗和业务消息还有data
     *
     * @param subCode 业务状态吗
     * @param subMsg  业务消息
     * @param d       返回的data
     * @param <T>     泛型
     * @return
     */
    public static <T> ResultVO<T> fail(String subCode, String subMsg, T d) {
        return ResultVO.<T>builder()
                .code(ResultCodeEnum.FAIL.getCode())
                .msg(ResultCodeEnum.FAIL.getMsg())
                .subCode(subCode)
                .subMsg(subMsg)
                .data(d)
                .timestamp(System.currentTimeMillis())
                .build();
    }

    /**
     * 短路的构建
     *
     * @param subCode
     * @param subMsg
     * @return
     */
    public static <T> ResultVO<T> fallback(String subCode, String subMsg) {
        return ResultVO.<T>builder()
                .code(ResultCodeEnum.FALLBACK.getCode())
                .msg(ResultCodeEnum.FALLBACK.getMsg())
                .subCode(subCode)
                .subMsg(subMsg)
                .timestamp(System.currentTimeMillis())
                .build();
    }

    /**
     * 无权限的构建
     *
     * @return
     */
    public static <T> ResultVO<T> unauthorized() {
        return unauthorized(null, null);
    }

    /**
     * 无权限的构建
     *
     * @param subCode
     * @param subMsg
     * @return
     */
    public static <T> ResultVO<T> unauthorized(String subCode, String subMsg) {
        return ResultVO.<T>builder()
                .code(ResultCodeEnum.UNAUTHORIZED.getCode())
                .msg(ResultCodeEnum.UNAUTHORIZED.getMsg())
                .subCode(subCode)
                .subMsg(subMsg)
                .timestamp(System.currentTimeMillis())
                .build();
    }

    /**
     * 格式校验失败返回的构建
     *
     * @param verificationFailedMsgList
     * @return
     */
    public static ResultVO verificationFailed(List<VerificationFailedMsg> verificationFailedMsgList) {
        return ResultVO.builder()
                .code(ResultCodeEnum.VERIFICATION_FAILED.getCode())
                .msg(ResultCodeEnum.VERIFICATION_FAILED.getMsg())
                .data(verificationFailedMsgList)
                .timestamp(System.currentTimeMillis())
                .build();
    }

    /**
     * 没有这个资源返回的构建
     *
     * @return
     */
    public static <T> ResultVO<T> notfound() {
        return ResultVO.<T>builder()
                .code(ResultCodeEnum.NOT_FOUND.getCode())
                .msg(ResultCodeEnum.NOT_FOUND.getMsg())
                .timestamp(System.currentTimeMillis())
                .build();
    }

    /**
     * 判断是否成功
     *
     * @return
     */
    public boolean isSuccess() {
        return this.code.equals(ResultCodeEnum.SUCCESS.getCode()) && this.subCode.equals(ResultCodeEnum.SUCCESS.getCode());
    }

    /**
     * 判断是否为失败
     *
     * @return
     */
    public boolean isFail() {
        return !this.isSuccess();
    }

    /**
     * 验证一个ResultVO是否成功
     *
     * @param r
     * @return
     */
    public static boolean isSuccess(ResultVO r) {
        return r.code.equals(ResultCodeEnum.SUCCESS.getCode()) && r.subCode.equals(ResultCodeEnum.SUCCESS.getCode());
    }

    /**
     * 验证一个ResultVO是否失败
     *
     * @param r
     * @return
     */
    public static boolean isFail(ResultVO r) {
        return !isSuccess(r);
    }

    /**
     * 判断是否为 ResultVO 类型
     *
     * @param other
     * @return
     */
    protected boolean canEqual(Object other) {
        return other instanceof ResultVO;
    }

}

