package cn.structure.common.entity;

import cn.structure.common.enums.ExceptionRsType;
import cn.structure.common.enums.ResultCodeEnum;
import cn.structure.common.exception.CommonException;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.io.Serializable;


/**
 * <p>
 * 出参：返回结果 - VO
 * 对单体比较友好也可以支持微服务
 * </p>
 *
 * @author chuck
 * @version 1.0.1
 * @since 2020/12/26 21:32
 */
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
@ApiModel(description = "出参：返回结果 - VO")
public class ResResultVO<T> implements Serializable {

    /**
     * 业务状态码
     */
    @ApiModelProperty(value = "业务状态码", example = "SUCCESS")
    private String code;

    /**
     * 返回的消息内容
     */
    @ApiModelProperty(value = "返回的消息内容", example = "成功！")
    private String message;

    /**
     * 业务是否成功
     */
    @ApiModelProperty(value = "业务是否成功", example = "true")
    private Boolean success;

    /**
     * 返回的数据
     */
    @ApiModelProperty(value = "返回的数据", example = "{}")
    private T data;

    /**
     * 系统响应的时间戳
     */
    @ApiModelProperty(value = "系统响应的时间戳", example = "112345644446")
    private Long timestamp;

    /**
     * 构建一个成功的ResResultVO
     *
     * @param t
     * @param <T>
     * @return
     */
    public static <T> ResResultVO<T> success(T t) {
        return ResResultVO.<T>builder()
                .code(ResultCodeEnum.SUCCESS.getCode())
                .message(ResultCodeEnum.SUCCESS.getMsg())
                .success(true)
                .timestamp(System.currentTimeMillis())
                .data(t)
                .build();
    }

    /**
     * 构建一个业务失败的ResResultVO
     *
     * @param code
     * @param message
     * @param <T>
     * @return
     */
    public static <T> ResResultVO<T> fail(String code, String message) {
        return ResResultVO.<T>builder()
                .code(code)
                .message(message)
                .success(false)
                .timestamp(System.currentTimeMillis())
                .build();
    }

    /**
     * 构建一个业务失败的带有出参结果的ResResultVO
     *
     * @param code    错误码
     * @param message 错误消息
     * @param t       返回数据
     * @param <T>     数据类型
     * @return
     */
    public static <T> ResResultVO<T> fail(String code, String message, T t) {
        return ResResultVO.<T>builder()
                .code(code)
                .message(message)
                .data(t)
                .success(false)
                .timestamp(System.currentTimeMillis())
                .build();
    }

    /**
     * 构建一个已知异常的出参
     *
     * @param code 错误码
     * @param message 错误消息
     * @return
     */
    public static <T> ResResultVO<T> exception(String code, String message) {
        return fail(code, message);
    }

    /**
     * 构建一个CommonException异常的出参 用于业务异常的捕获
     *
     * @param ce 异常内容
     * @return
     */
    public static <T> ResResultVO<T> exception(CommonException ce) {
        return fail(ce.getCode(), ce.getMessage());
    }

    /**
     * 构建一个未知异常的出参
     *
     * @param <T>
     * @return
     */
    public static <T> ResResultVO<T> exception() {
        return exception(ResultCodeEnum.ERR.getCode(),ResultCodeEnum.ERR.getMsg());
    }

    /**
     * 构建一个未知异常需要输出消息提示的的出参
     *
     * @param message
     * @param <T>
     * @return
     */
    public static <T> ResResultVO<T> exception(String message) {
        return exception(ResultCodeEnum.ERR.getCode(),message);
    }

    /**
     * build 一个失败的返回结果
     *
     * @param resultCodeEnum
     * @return
     */
    public static <T> ResResultVO<T> build(ResultCodeEnum resultCodeEnum) {
        return fail(resultCodeEnum.getCode(), resultCodeEnum.getMsg());
    }

    /**
     * build 一个没有权限的返回结果
     *
     * @param exceptionRsType
     * @return
     */
    public static <T> ResResultVO<T> build(ExceptionRsType exceptionRsType) {
        return fail(exceptionRsType.getCode(), exceptionRsType.getMsg());
    }
}
