package cn.springlet.redis.config;

import com.alibaba.fastjson.support.spring.GenericFastJsonRedisSerializer;
import org.springframework.cache.CacheManager;
import org.springframework.cache.annotation.CachingConfigurerSupport;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.cache.RedisCacheWriter;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.script.DefaultRedisScript;
import org.springframework.data.redis.serializer.RedisSerializationContext;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.time.Duration;

import static java.util.Collections.singletonMap;

/**
 * redis配置
 */
@Configuration
@EnableCaching
public class SpringletRedisConfig extends CachingConfigurerSupport {


    @Bean("redisNoNewScript")
    public DefaultRedisScript<Long> redisNoNewScript() {
        DefaultRedisScript<Long> redisScript = new DefaultRedisScript<>();
        redisScript.setScriptText("if redis.call('get', KEYS[1]) then \n" +
                "    return 1 \n" +
                "else \n" +
                "    redis.call('set', KEYS[1], ARGV[1]) \n" +
                "    redis.call('pexpire', KEYS[1], ARGV[2]) \n" +
                "    return 0 \n" +
                "end");
        redisScript.setResultType(Long.class);
        return redisScript;
    }

    @Bean("redisRateLimiterScript")
    public DefaultRedisScript<Long> redisRateLimiterScript() {
        DefaultRedisScript<Long> redisScript = new DefaultRedisScript<>();
        //两个循环，第一个判断是否超限，如果对应规则的索引，否则进入第二个循环，将所有规则的值自增
        redisScript.setScriptText("for i=1,#KEYS do\n" +
                " local current = redis.call('get', KEYS[i])\n" +
                " if current and tonumber(current) >= tonumber(ARGV[i*2-1]) then\n" +
                "  return i-1\n" +
                " end\n" +
                "end\n" +
                "\n" +
                "for i=1,#KEYS do\n" +
                "    local current = redis.call('incr', KEYS[i])\n" +
                "    if tonumber(current) == 1 then\n" +
                "        redis.call('pexpire', KEYS[i], ARGV[i*2])\n" +
                "    end\n" +
                "end");
        redisScript.setResultType(Long.class);
        return redisScript;
    }

    /**
     * @param connectionFactory
     * @return
     */
    @Bean
    @SuppressWarnings(value = {"all"})
    public RedisTemplate<Object, Object> redisTemplate(RedisConnectionFactory connectionFactory) {
        RedisTemplate<Object, Object> template = new RedisTemplate<>();
        template.setConnectionFactory(connectionFactory);

        GenericFastJsonRedisSerializer serializer = new GenericFastJsonRedisSerializer();
        StringRedisSerializer keySerializer = new StringRedisSerializer();

        // 使用StringRedisSerializer来序列化和反序列化redis的key值
        template.setKeySerializer(keySerializer);
        template.setValueSerializer(serializer);

        // Hash的key也采用StringRedisSerializer的序列化方式
        template.setHashKeySerializer(keySerializer);
        template.setHashValueSerializer(serializer);

        template.afterPropertiesSet();
        return template;
    }


    @Bean
    @SuppressWarnings(value = {"all"})
    public CacheManager cacheManager(RedisConnectionFactory factory, RedisCacheProperties redisCacheProperties) {
        // 配置序列化（缓存默认有效期 6小时）
        RedisCacheConfiguration config = RedisCacheConfiguration.defaultCacheConfig()
                .computePrefixWith(name -> name + "")
                .entryTtl(Duration.ofSeconds(redisCacheProperties.getTimeout()));
        RedisCacheConfiguration redisCacheConfiguration = config.serializeKeysWith(RedisSerializationContext.SerializationPair.fromSerializer(new StringRedisSerializer()))
                .serializeValuesWith(RedisSerializationContext.SerializationPair.fromSerializer(new GenericFastJsonRedisSerializer()));

        RedisCacheManager cacheManager = RedisCacheManager.builder(RedisCacheWriter.lockingRedisCacheWriter(factory)).cacheDefaults(redisCacheConfiguration)
                .withInitialCacheConfigurations(singletonMap("test_cache", RedisCacheConfiguration.defaultCacheConfig().entryTtl(Duration.ofMinutes(5)).disableCachingNullValues()))
                .transactionAware().build();
        return cacheManager;
    }
}
