package cn.sowjz.search.core.doc;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * 专门负责读取格式化文件。该类具有一个缓冲区。
 * <p>
 * 格式化文件的内容是多个doc。而且，格式化文件是2进制的形式。里面的存储形式是doc转化成为ByteBuffer后的形式。
 * </p>
 * @author Ryan
 * @since 1.2
 * @version 1.1
 * @date 2007-4-14
 */
public class DocReader
{
	private byte[] buffer;


	private int used;

	/**
	 * 默认缓冲区大小
	 */
	public static final int LEN = (1 << 20);

	/**
	 * 构造，默认的缓冲区大小是<code>DocReader.LEN</code>
	 */
	public DocReader()
	{
		buffer = new byte[LEN];
		used = 0;
	}

	/**
	 * 得到缓冲 -- 缓冲区
	 * @return 缓冲区
	 */
	public byte[] getBuffer()
	{
		return buffer;
	}

	/**
	 * 设置一个缓冲区
	 * @param buffer -- 要设置的缓冲区
	 */
	public void setBuffer(byte[] buffer)
	{
		this.buffer = buffer;
	}

	/**
	 * 得到缓冲区已经使用的大小
	 * @return 缓冲区使用的大小
	 */
	public int getUsed()
	{
		return used;
	}

	/**
	 * 读格式化文件
	 * @param f -- 格式化文件
	 * @return -- 是否读取了格式化文件的内容
	 * @throws IOException -- 读文件时出错抛出
	 */
	public boolean read(File f) throws IOException
	{
		if (!f.exists())
			return false;
		long len = f.length();
		if (used + len > LEN)
			return false;
		FileInputStream fin = new FileInputStream(f);
		fin.read(buffer, used, (int) len);
		used += len;

		closeStream(fin, null);

		return true;
	}

	/**
	 * 清空缓冲区
	 */
	public void clearBuf()
	{
		used = 0;
	}

	/**
	 * 关闭流
	 * @param in
	 * @param out
	 */
	private void closeStream(InputStream in, OutputStream out)
	{
		if (in != null)
		{
			try
			{
				in.close();
			} catch (Exception ex)
			{
				// ignore
			} finally
			{
				if (in != null)
					in = null;
			}
		}
		if (out != null)
		{
			try
			{
				out.close();
			} catch (Exception ex)
			{
				// ignore
			} finally
			{
				if (out != null)
					out = null;
			}
		}
	}


}
