package cn.sowjz.search.core.db;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import cn.sowjz.search.common.ByteBuff;
import cn.sowjz.search.common.VConvert;
import cn.sowjz.search.core.Constants;


/**
 * 全文库结构的描述
 */
public class Schema
{
	private List<FieldInfo> infos;

	private int sortno;
	
	private int seqno;

	/**
	 * 构造
	 */
	public Schema()
	{
		this(6);
	}

	/**
	 * 构造
	 * @param size -- 大小
	 */
	public Schema(int size)
	{
		sortno = -1;
		seqno=-1;
		infos = new ArrayList<FieldInfo>(size);
	}

	/**
	 * 字段个数
	 * @return 字段个数
	 */
	public int fieldNum()
	{
		return infos.size();
	}

	/**
	 * 添加一个字段
	 * @param info
	 */
	public void add(FieldInfo info)
	{
		infos.add(info);
	}

	/**
	 * 添加ID字段
	 */
	public void addIDField()
	{
		FieldInfo info = new FieldInfo(infos.size(), "ID", FieldInfo.TYPE_SEQUENCE, 1, false, false, true, true, false, true);
		infos.add(info);
	}

	/**
	 * 通过字段名，查找字段
	 * @param name -- 字段名
	 * @return 字段
	 */
	public FieldInfo find(String name)
	{
		
		if(name.endsWith(Constants.HostSnFieldName))
			return new FieldInfo(-1, name, FieldInfo.TYPE_HOSTSN, 1,false,false,true,false,false,false);
		
		if(name.endsWith("01"))
			return new FieldInfo(-1, name, FieldInfo.TYPE_TEXT, 1,false,false,true,false,false,false);
		if(name.endsWith("02"))
			return new FieldInfo(-1, name, FieldInfo.TYPE_BYTE16, 1,false,false,true,false,false,false);
		for (FieldInfo info : infos)
		{
			if (info.getName().equals(name))
				return info;
		}
		return null;
	}

	/**
	 * 通过序号，查找字段
	 * @param sn
	 * @return 字段
	 */
	public FieldInfo find(int sn)
	{
		for (FieldInfo info : infos)
		{
			if (info.getSn() == sn)
				return info;
		}
		return null;
	}

	/**
	 * 得到所有的字段信息
	 * @return 所有字段的集合
	 */
	public List<FieldInfo> getInfos()
	{
		return infos;
	}

	/**
	 * 得到第i个字段
	 * @param i
	 * @return 字段
	 */
	public FieldInfo get(int i)
	{
		if (i < 0 || fieldNum() == 0 || i >= fieldNum())
			return null;
		else
			return infos.get(i);
	}

	/**
	 * 得到排序字段的序号
	 * @return 排序字段序号
	 */
	public int getSortno()
	{
		return sortno;
	}

	/**
	 * 设置排序字段的序号
	 * @param sortno
	 */
	public void setSortno(int sortno)
	{
		this.sortno = sortno;
	}
	
	public void setSortnoByName(String fieldname) throws Exception
	{
		int n=0;
		for (FieldInfo info : getInfos())
			info.setSn(n++);
		
		FieldInfo fi = find(fieldname);
		if(fi==null)throw new Exception("no such a field:"+fieldname);
		if(fi.getType()!=FieldInfo.TYPE_INT32 && fi.getType()!=FieldInfo.TYPE_INT64)
			throw new Exception("the type of sort field must be TYPE_INT32 or TYPE_INT64");
		this.sortno = fi.getSn();
	}
	
	public FieldInfo getSeqFI()
	{
		if(seqno==-1)return null;
		if(infos==null)return null;
		if(seqno>=infos.size())return null;
		return infos.get(seqno);
	}

	@Override
	public String toString()
	{
		StringBuffer buf = new StringBuffer(256);
		buf.append("size:").append(fieldNum()).append(", sortno:").append(sortno).append("\n");
		for (int i = 0; i < fieldNum(); i++)
			buf.append(i).append("====> ").append(get(i).toString()).append("\n");
		return buf.toString();
	}

	/**
	 * 把自己转化成字节流，便于网络传输，网络传输的时候会自动调用。
	 * @param charset -- 字符集
	 * @return 字节流对象
	 * @throws IOException
	 */
	public ByteBuff toByteBuffer(String charset) throws IOException
	{
		ByteBuff buf = new ByteBuff();
		// num (c:char)
		buf.append((byte) fieldNum());
		// info
		for (FieldInfo info : infos)
			buf.append(info.toByteBuffer(charset));
		// sortno(c:char)
		return buf.append((byte) sortno);
	}

	/**
	 * 把字节流转化成自己
	 * @param buf -- 字节数组
	 * @param charset -- 字符集
	 * @return FieldInfos
	 * @throws IOException
	 */
	public static Schema bytes2Me(byte[] buf, String charset) throws IOException
	{
		int i = 0;
		int k = 0;
		byte[] b = new byte[7];
		Schema schema = new Schema(buf[i++]);
		for (int j = 0; j < (int) buf[0]; j++)
		{
			FieldInfo info = new FieldInfo();
			info.setSn((int) buf[i++]);
			for (k = 0; k < 6; k++)
				b[k] = buf[i++];
/*			
			for (k = 0; k < 7; k++)
			{
				if (b[k] == 0)
					break;
			}
*/			
			info.setName(new String(b, 0, 2, charset));
			// info.setName(new String(b, 0, 2, charset));
			info.setFlag( buf[i++]);
			info.setType((int) buf[i++]);
			if(info.getType()==FieldInfo.TYPE_SEQUENCE)
				schema.seqno=j;
			info.setWeight((int) buf[i++]);
			info.setCached(VConvert.byte2Bool(buf[i++]));
			info.setIndex(VConvert.byte2Bool(buf[i++]));
			info.setCompare(VConvert.byte2Bool(buf[i++]));
			info.setUnsign(VConvert.byte2Bool(buf[i++]));
			info.setTime(VConvert.byte2Bool(buf[i++]));
			info.setStore(VConvert.byte2Bool(buf[i++]));
			schema.add(info);
		}
		schema.setSortno((int) buf[i]);
		return schema;
	}

	public int howManySequenceField()
	{
		int num = 0;
		for (int i = 0; i < fieldNum(); i++)
		{
			if (infos.get(i).getType() == FieldInfo.TYPE_SEQUENCE)
				num++;
		}
		return num;
	}

	public String getAllNames() {
		StringBuffer buf = new StringBuffer(256);
		
		for (int i = 0; i < fieldNum(); i++)
			buf.append(get(i).getName()).append(",");
		return buf.toString();
	}
}
