package cn.sowjz.search.core.db;

import java.io.IOException;

import cn.sowjz.search.common.ByteBuff;
import cn.sowjz.search.core.Constants;

/**
 * 描述一个全文库字段的类
 * @author Ryan
 * @since 1.0
 * @version 1.1
 */
public class FieldInfo
{
	
	public static final int TYPE_HOSTSN=0;
	public static final int TYPE_TEXT=1;
	public static final int TYPE_VARCHAR=2;
	public static final int TYPE_INT32=3;
	public static final int TYPE_INT64=4;
	public static final int TYPE_CATEGORY=5;
	public static final int TYPE_SEQUENCE=6;
	public static final int TYPE_ARTICLE=7;
	public static final int TYPE_BYTE16=8;
	public static final int TYPE_CLOB=9;
	public static final int TYPE_BINARY=10;
	public static final int TYPE_BYTE=11;
	public static final int TYPE_SHORT=12;
	public static final int TYPE_BIT=13;
	public static final int TYPE_BIT2=14;
	public static final int TYPE_BIT4=15;
	public static final int TYPE_KWORDS=16;
	public static final int TYPE_CATEVINT=17;
	public static final int TYPE_INT24=18;
	public static final int TYPE_LABEL=19;
	public static final int TYPE_INT32I=20;
	
	public static final String types[]=new String[]{"TYPE_TEXT","TYPE_VARCHAR","TYPE_INT32","TYPE_INT64","TYPE_CATEGORY"
		,"TYPE_SEQUENCE","TYPE_ARTICLE","TYPE_BYTE16","TYPE_CLOB","TYPE_BINARY"
		,"TYPE_BYTE","TYPE_SHORT","TYPE_BIT","TYPE_BIT2","TYPE_BIT4"
		,"TYPE_KWORDS","TYPE_CATEVINT","TYPE_INT24","TYPE_LABEL","TYPE_INT32I"};
	
	private String name;

	private int sn; // serial number

	private int type;

	private int weight;

	private boolean cached;

	private boolean index;

	private boolean compare;

	private boolean unsign;

	private boolean time;

	private boolean store;
	protected  byte flag;

	/**
	 * 构造
	 */
	public FieldInfo()
	{
		weight=1;
		store=true;
	}

	/**
	 * 构造
	 * @param name -- 字段名
	 * @param type -- 类型
	 * @param weight -- 权重
	 * @param cached -- 是否被缓存
	 * @param index -- 是否分词
	 * @param compare -- 是否比较
	 * @param unsign -- 是否带符号
	 * @param time -- 是否是时间
	 * @param store -- 是否存储
	 */
	public FieldInfo(int sn, String name, int type,int flag, int weight, boolean cached, boolean index, boolean compare, boolean unsign, boolean time, boolean store)
	{
		this.name = name;
		this.sn = sn; // serial number
		this.type = type;
		this.weight = weight;
		this.cached = cached;
		this.index = index;
		this.compare = compare;
		this.unsign = unsign;
		this.time = time;
		this.store = store;
		this.flag=(byte)flag;
	}
	public FieldInfo(int sn, String name, int type, int weight, boolean cached, boolean index, boolean compare, boolean unsign, boolean time, boolean store)
	{
		this(sn,name,type,0,weight,cached, index, compare, unsign, time, store);
	}
	public FieldInfo(int sn, String name, int type, int weight, boolean cached,  boolean unsign, boolean time, boolean store)
	{
		this(sn,name,type,0,weight,cached, false, false, unsign, time, store);
	}
	public FieldInfo(byte[] bb)
	{
		sn=bb[0]&0xff;
		name=new String(bb,1,2);
		
		 type=bb[8]&0xff;
		 weight=bb[9]&0xff;
		 cached=bb[10]==1;
		 index=bb[11]==1;
         compare=bb[12]==1;
         unsign=bb[13]==1;
         time=bb[14]==1;
         store=bb[15]==1;
         flag=bb[7];
	}

	/**
	 * 是否压缩
	 * @return true表示是。false表示不是
	 */
	public boolean isCached()
	{
		return cached;
	}

	/**
	 * 设置压缩
	 * @param cached
	 */
	public void setCached(boolean cached)
	{
		this.cached = cached;
	}

	/**
	 * 是否分词
	 * @return true表示是。false表示不是
	 */
	public boolean isIndex()
	{
		return index;
	}

	/**
	 * 设置分词
	 * @param index
	 */
	public void setIndex(boolean index)
	{
		this.index = index;
	}

	/**
	 * 得到字段名
	 * @return 字段名
	 */
	public String getName()
	{
		return name;
	}

	/* used in  BaseRequest.toString()*/
	public String getName0()
	{
		if(name!=null && name.equals(Constants.HostSnFieldName))
			return "host_sn";
		return name;
	}
	
	/**
	 * 设置字段名
	 * @param name
	 */
	public void setName(String name)
	{
		this.name = name;
	}

	/**
	 * 返回序号
	 * @return 字段序号
	 */
	public int getSn()
	{
		return sn;
	}

	/**
	 * 设置序号
	 * @param sn
	 */
	public void setSn(int sn)
	{
		this.sn = sn;
	}

	/**
	 * 是否存储
	 * @return true表示是。false表示不是
	 */
	public boolean isStore()
	{
		return store;
	}

	/**
	 * 设置存储
	 * @param store
	 */
	public void setStore(boolean store)
	{
		this.store = store;
	}

	/**
	 * 是否是时间
	 * @return true表示是。false表示不是
	 */
	public boolean isTime()
	{
		return time;
	}

	/**
	 * 设置是否是时间
	 * @param time
	 */
	public void setTime(boolean time)
	{
		this.time = time;
	}

	/**
	 * 是否是Sequenece字段
	 * @return true表示是。false表示不是
	 */
	public boolean isSequenceField()
	{
		return type == TYPE_SEQUENCE;
	}

	/**
	 * 是否是文本字段
	 * @return true表示是。false表示不是
	 */
	public boolean isTextField()
	{
		return type == TYPE_TEXT || type==TYPE_ARTICLE;
	}

	public boolean isVarcharField()
	{
		return type==TYPE_VARCHAR ||type==TYPE_LABEL;
	}
	public boolean isBinaryField()
	{
		return type==TYPE_BINARY;
	}
	public boolean isClobField()
	{
		return type==TYPE_CLOB;
	}
	public boolean isArticleField()
	{
		return type==TYPE_ARTICLE;
	}
	public boolean isByte16Field()
	{
		return type==TYPE_BYTE16;
	}
	public boolean isNativeID()
	{
		return type==TYPE_SEQUENCE && flag==0;
	}
	/**
	 * 是否是分类字段
	 * @return true表示是。false表示不是
	 */
	public boolean isCateField()
	{
		return type == TYPE_CATEGORY||type == TYPE_CATEVINT;
	}

	/**
	 * 是否是int32字段
	 * @return true表示是。false表示不是
	 */
	public boolean isInt32Field()
	{
		return type == TYPE_INT32||type == TYPE_INT32I;
	}
	public boolean isInt32TimeField()
	{
		return type == TYPE_INT32 && time;
	}

	public boolean isNumberField(){
		return isIntField()|| isInt64Field()||isBitField();
	}
	public boolean isIntField()
	{
    return type==TYPE_INT32 || type==TYPE_BYTE || type==TYPE_SHORT|| type==TYPE_INT24|| type==TYPE_HOSTSN||type == TYPE_INT32I;
 	}
 	public boolean isBitField()
 	{
 		return type==TYPE_BIT || type==TYPE_BIT2 || type==TYPE_BIT4;
 		}
	/**
	 * 是否是int64字段
	 * @return true表示是。false表示不是
	 */
	public boolean isInt64Field()
	{
		return type == TYPE_INT64 ||type==TYPE_SEQUENCE;
	}

	public boolean isStringField()
	{
		return type==TYPE_TEXT|| type==TYPE_VARCHAR || type==TYPE_CATEGORY|| type==TYPE_CATEVINT|| type==TYPE_ARTICLE
		         || type==TYPE_CLOB||type==TYPE_KWORDS||type==TYPE_LABEL;

	}
	/**
	 * 设置字段类型
	 * @param type -- 字段类型
	 */
	public void setType(int type)
	{
		this.type = type;
	}

	/**
	 * 是否有符号
	 * @return true表示是。false表示不是
	 */
	public boolean isUnsign()
	{
		return unsign;
	}

	/**
	 * 设置符号
	 * @param unsign
	 */
	public void setUnsign(boolean unsign)
	{
		this.unsign = unsign;
	}

	/**
	 * 得到权重
	 * @return 字段权重
	 */
	public int getWeight()
	{
		return weight;
	}

	/**
	 * 设置权重
	 * @param weight
	 */
	public void setWeight(int weight)
	{
		this.weight = weight;
	}

	/**
	 * 是否比较
	 * @return true表示是。false表示不是
	 */
	public boolean isCompare()
	{
		return compare || isCateField();
	}

	/**
	 * 设置比较
	 * @param compare
	 */
	public void setCompare(boolean compare)
	{
		this.compare = compare;
	}

	/**
	 * 得到字段类型
	 * @return 字段类型
	 */
	public int getType()
	{
		return type;
	}

	public byte getFlag()
	{
		return flag;
	}
	public void setFlag(byte f)
	{
		flag=f;
	}
	@Override
	public String toString()
	{
		StringBuffer buf = new StringBuffer(20);
		buf.append("name:").append(name).append(", sn:").append(sn).append(", type:").append(type).append("|").append(types[type-1])
		.append(", flag:").append(flag)	.append(", weight:").append(weight);
		buf.append(", cached: ").append(cached).append(", index:").append(index).append(", compare:").append(compare).append(", unsign:").append(unsign);
		buf.append(", time:").append(time).append(", store:").append(store);
		return buf.toString();
	}

	/**
	 * 转化成为字节流，便于网络传输。在进行网络传输的时候，会自动地通过此方法进行传输
	 * @param charset -- 字符集
	 * @return 字节流对象
	 * @throws IOException
	 */
	public ByteBuff toByteBuffer(String charset) throws IOException
	{
		ByteBuff buf = new ByteBuff();
		buf.append((byte) sn); // sn (c:char)
		byte[] namebytes = null;
		namebytes = name.getBytes(charset);
		buf.append(namebytes, 0, namebytes.length); // name (char[7])
		for (int j = 0; j < 6 - namebytes.length; j++)
			buf.append((byte) 0);
		// type(c:char), weight(c:char), cached(c:char), index(c:char),compare(c:char),unsigned(c:char),time(c:char),store(c:char)
		buf.append((byte) flag);
		buf.append((byte) type).append((byte) weight).append(cached).append(index).append(compare).append(unsign).append(time).append(store);
		return buf;
	}
	
	public String getTypeName()
	{
		
		switch(type)
		{
		     case TYPE_TEXT: return "正文";
		     case TYPE_VARCHAR : return "短语";
		     case TYPE_BINARY:  return "存储";
		     case TYPE_INT32: return "int32";
		     case TYPE_INT64 :return "int64";
		     case TYPE_CATEGORY : return "分类";
		     case TYPE_CATEVINT : return "分类";
		     case TYPE_SEQUENCE : return "序列";
		     case TYPE_ARTICLE : return "文章";
		     case TYPE_BYTE16 : return "byte16";
		     case TYPE_CLOB : return "clob";
		     case TYPE_BYTE : return "byte";
		     case TYPE_SHORT :return "short";
		     case TYPE_BIT : return "bit";
		     case TYPE_BIT2 : return "bit2";
		     case TYPE_BIT4 : return "bit4";
		     case TYPE_KWORDS : return "关键词";
		     case TYPE_INT24 :return "int24";
		     case TYPE_LABEL : return "标签";
		     case TYPE_INT32I: return "int32i";
		}
		 return "未知";
	}
	
	
	public FieldInfo sn(int sn){
		this.sn=sn;
		return this;
	}
	public FieldInfo name( String name){
		this.name=name;
		return this;
	}
	public FieldInfo type( int type){
		this.type=type;
		return this;
	}
	public FieldInfo weight( int weight){
		this.weight=weight;
		return this;
	} 
	public FieldInfo cached( boolean cached){
		this.cached=cached;
		return this;
	} 
	public FieldInfo index(boolean index){
		this.index=index;
		return this;
	} 
	public FieldInfo compare(boolean compare){
		this.compare=compare;
		return this;
	} 
	public FieldInfo unsign(boolean unsign){
		this.unsign=unsign;
		return this;
	}
	public FieldInfo time(boolean time){
		this.time=time;
		return this;
	} 
	public FieldInfo store(boolean store){
		this.store=store;
		return this;
	}
	
	public boolean supportHtml(){
		return (type == TYPE_TEXT || type==TYPE_ARTICLE) && (flag &2)>0;
	}
	
	public static class Text extends FieldInfo{
		public Text(String name){
			super();
			name(name);
			type(TYPE_TEXT);
		}
	}
	
	public static class Varchar extends FieldInfo{
		public Varchar(String name){
			super();
			name(name);
			type( TYPE_VARCHAR);
		}
	}

	public static class Int32 extends FieldInfo{
		public Int32(String name){
			super();
			name(name);
			type( TYPE_INT32);
		}
	}
	public static class Int64 extends FieldInfo{
		public Int64(String name){
			super();
			name(name);
			type( TYPE_INT64);
		}
	}
	public static class Category extends FieldInfo{
		public Category(String name){
			super();
			name(name);
			type( TYPE_CATEGORY);
		}
	}
	public static class Sequence extends FieldInfo{
		public Sequence(String name){
			super();
			name(name);
			type( TYPE_SEQUENCE);
		}
		public Sequence external(boolean external){
			this.flag=(byte) (external?1:0);
			return this;
		}
	}
	public static class Article extends FieldInfo{
		public Article(String name){
			super();
			name(name);
			type( TYPE_ARTICLE);
			flag=1;
		}
		public Article html(boolean html){
			this.flag=(byte) (html?3:1);
			return this;
		}
	}
	public static class Byte16 extends FieldInfo{
		public Byte16(String name){
			super();
			name(name);
			type( TYPE_BYTE16);
		}
		
		public Byte16 supportDistinct(){
			flag=1;
			return this;
		}
	}
	public static class Clob extends FieldInfo{
		public Clob(String name){
			super();
			name(name);
			type( TYPE_CLOB);
		}
	}
	public static class Binary extends FieldInfo{
		public Binary(String name){
			super();
			name(name);
			type(TYPE_BINARY);
		}
	}
	public static class Byte extends FieldInfo{
		public Byte(String name){
			super();
			name(name);
			type( TYPE_BYTE);
		}
	}
	public static class Short extends FieldInfo{
		public Short(String name){
			super();
			name(name);
			type( TYPE_SHORT);
		}
	}
	public static class Bit extends FieldInfo{
		public Bit(String name){
			super();
			name(name);
			type( TYPE_BIT);
		}
	}
	public static class Bit2 extends FieldInfo{
		public Bit2(String name){
			super();
			name(name);
			type( TYPE_BIT2);
		}
	}
	public static class Bit4 extends FieldInfo{
		public Bit4(String name){
			super();
			name(name);
			type( TYPE_BIT4);
		}
	}
	public static class Kwords extends FieldInfo{
		public Kwords(String name){
			super();
			name(name);
			type(TYPE_KWORDS);
			compare(true);
		}
	}
	public static class CateVint extends FieldInfo{
		public CateVint(String name){
			super();
			name(name);
			type( TYPE_CATEVINT);
		}
	}
	public static class Int24 extends FieldInfo{
		public Int24(String name){
			super();
			name(name);
			type( TYPE_INT24);
		}
	}
	public static class Label extends FieldInfo{
		public Label(String name){
			super();
			name(name);
			type( TYPE_LABEL);
		}
	}

	public static class Int32I extends FieldInfo{
		public Int32I(String name){
			super();
			name(name);
			type( TYPE_INT32I);
		}
	}
}
