package cn.sowjz.search.core;



import java.util.ArrayList;
import java.util.List;

import cn.sowjz.search.common.ByteBuff;
import cn.sowjz.search.common.util.StringUtil;
import cn.sowjz.search.core.conf.SearchConfig;
import cn.sowjz.search.core.conf.SpecialIdxParser;
import cn.sowjz.search.core.db.FieldInfo;
import cn.sowjz.search.core.db.Schema;
import cn.sowjz.search.core.doc.Doc;
import cn.sowjz.search.core.exception.DBExistException;
import cn.sowjz.search.core.exception.NoDBException;
import cn.sowjz.search.core.exception.SysConnectionException;
import cn.sowjz.search.core.net.control.AskIndex;
import cn.sowjz.search.core.net.control.AskMining;
import cn.sowjz.search.core.net.control.AskQuery;
import cn.sowjz.search.core.net.control.BaseConn;
import cn.sowjz.search.core.net.control.IndexConn;
import cn.sowjz.search.core.net.control.QueryConn;
import cn.sowjz.search.core.net.pool.ConnPool;
import cn.sowjz.search.core.net.pool.ISearchConnPool;
import cn.sowjz.search.core.net.pool.IndexConnBuilder;
import cn.sowjz.search.core.net.pool.NoPool;
import cn.sowjz.search.core.net.pool.QueryConnBuilder;
import cn.sowjz.search.core.query.request.BaseRequest;
import cn.sowjz.search.core.query.request.ClusterRequest;
import cn.sowjz.search.core.query.request.CubeRequest;
import cn.sowjz.search.core.query.request.DistinctRequest;
import cn.sowjz.search.core.query.request.GroupRequest;
import cn.sowjz.search.core.query.request.KeyWordRequest;
import cn.sowjz.search.core.query.request.QueryRequest;
import cn.sowjz.search.core.query.request.UnitedRequest;
import cn.sowjz.search.core.query.request.WamRequest;
import cn.sowjz.search.core.query.request.WordCloudRequest;
import cn.sowjz.search.core.query.response.AsynInfo;
import cn.sowjz.search.core.query.response.ClusterResponse;
import cn.sowjz.search.core.query.response.CubeResponse;
import cn.sowjz.search.core.query.response.DbSeqScope;
import cn.sowjz.search.core.query.response.DistinctResponse;
import cn.sowjz.search.core.query.response.ExamineResponse;
import cn.sowjz.search.core.query.response.GroupResponse;
import cn.sowjz.search.core.query.response.QueryResponse;
import cn.sowjz.search.core.query.response.SummaryResponse;
import cn.sowjz.search.core.query.response.UnitedResponse;
import cn.sowjz.search.core.query.response.UpdateResponse;
import cn.sowjz.search.core.query.response.WamResponse;
import cn.sowjz.search.core.query.response.WordCloudResponse;
import cn.sowjz.search.core.query.response.XWord;
import cn.sowjz.search.core.server.state.ServerIdxThreadStates;
import cn.sowjz.search.core.server.state.ServerRegisterState;
import cn.sowjz.search.core.util.TxUtil;

/**
 */
public class SearchClient extends SearchBase
{
	public static boolean open_on_create=true;
	public final static String Api_Version="201";
	public final static int Isearch_Version_Support=196;
	public final static int Once_Feed_Max_Table=100;
	
	


	
	
	private ConnPool<QueryConn>  qconn_pool= null;
	private ConnPool<IndexConn>  iconn_pool= null;
	
	
	

	/**
	 * 构造方法
	 * @param cfg -- 配置信息
	 * @throws SysConnectionException
	 */
	public SearchClient(SearchConfig cfg) throws Exception
	{   
		
		this.cfg=cfg;
		
		init_ConnPool(cfg);
		
		if(open_on_create) open();
		sparser =  new SpecialIdxParser(cfg.getProps());
	}
	//for cache api
	protected SearchClient()
	{
			
	}
	
	protected void init_ConnPool(SearchConfig cfg) throws Exception
	{
		if(cfg.getPropertyOfboolean("system.net.sock.query.pool.enable"))
			qconn_pool=new ISearchConnPool<QueryConn>(new QueryConnBuilder(),
					cfg.getPropertyOfint("system.net.sock.query.pool.maxActive"),
					cfg.getPropertyOfint("system.net.sock.query.pool.maxIdle"),
					cfg.getPropertyOflong("system.net.sock.query.pool.maxWait.ms"),
					cfg.getPropertyOfint("system.net.sock.query.pool.idleRemoveDelay.second"),
					cfg.getPropertyOflong("system.net.sock.query.pool.pulse.second")*1000
					);
		else	
			qconn_pool=new NoPool<QueryConn>(new QueryConnBuilder());
		
		qconn_pool.initialize(cfg);
		
		if(cfg.getPropertyOfboolean("system.net.sock.index.pool.enable"))
			iconn_pool=new ISearchConnPool<IndexConn>(new IndexConnBuilder(),
					cfg.getPropertyOfint("system.net.sock.index.pool.maxActive"),
					cfg.getPropertyOfint("system.net.sock.index.pool.maxIdle"),
					cfg.getPropertyOflong("system.net.sock.index.pool.maxWait.ms"),
					cfg.getPropertyOfint("system.net.sock.index.pool.idleRemoveDelay.second"),
					cfg.getPropertyOflong("system.net.sock.index.pool.pulse.second")*1000
					);
		else
			iconn_pool=new NoPool<IndexConn>(new IndexConnBuilder());
		iconn_pool.initialize(cfg);
		
	}
	/**
	 * 打开全文检索系统，如果存在会得到全文库的配置结构
	 * @throws SysConnectionException -- 当网络不通时会抛出该异常
	 */
	 protected void open() throws Exception
	{
		log.info("Opening system .");
		QueryConn conn=qconn_pool.getConn();
		try
		{
			testServerConnection(conn);
		} catch (Exception ex)
		{
			qconn_pool.releaseConn(conn,ex);
			log.error("open system error : " + ex.getMessage());
			throw new SysConnectionException(ex.getMessage());
		}
		readFeedInfo(conn);
		
		try
		{
			schema = descDb(conn);
		} catch (Exception ex)
		{
			qconn_pool.releaseConn(conn,ex);
			log.warn("open system warn(descDb()): " + ex.getMessage());
			// Ingore;
			return;
		}
		
		
		qconn_pool.releaseConn(conn);
		log.info("System has been opened.");
		
	}


	protected void readFeedInfo(BaseConn conn)throws Exception {
		
		feedinfo=new AskQuery(conn).feedInfo();
		
		if(feedinfo==null)
			throw new Exception("read feed info from "+cfg.getSockIp()+":"+cfg.getQuerySockPort()+" failed");
		
		FEED_BUf_MAX_LEN=feedinfo.feedBuffSize;
		if(FEED_BUf_MAX_LEN<=0)
			throw new Exception("max feed size is error");
		log.info("max feed buffer size = "+(FEED_BUf_MAX_LEN>>10)+"KB");
		
		if(feedinfo.version % 1000<Isearch_Version_Support)
			throw new Exception("The isearch version["+feedinfo.version+"] is low, this API doesnt support. the version of isearch should be or above "+Isearch_Version_Support+".");
	}

	

	
	
	
	/**
	 * 查询注册信息
	 * @return -- 注册信息
	 * @throws Exception
	 */
	public ServerRegisterState queryRegisterState() throws Exception
	{
		ServerRegisterState regState=null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			regState=queryRegisterState(conn);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}		
		qconn_pool.releaseConn(conn);
		
		
		
		return regState;
	}
	public ServerRegisterState queryRegisterState(QueryConn conn) throws Exception
	{
		ServerRegisterState regState =new AskQuery(conn).queryRegisterState(feedinfo.getCharset());
		if(regState==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Query Server Register Info Successful .");
		return regState;
	}
	/**
	 * 测试服务器的连接状况
	 * @throws Exception
	 */
	public void testServerConnection() throws Exception
	{
		//log.info("Test Server Query Connection.");
		QueryConn conn=qconn_pool.getConn();
		try
		{
			testServerConnection(conn);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
	}
	@Override
	public void testServerConnection(BaseConn conn) throws Exception
	{
		//log.info("Test Server Query Connection.");
		if(new AskQuery(conn).testConnection())
			log.info("Test Server Query Connection Successful .");
		else 
			throw new Exception("Test Server Query Connection failed! "+conn.getErrMsg());
			
	}
	/**
	 * 测试服务器加载线程的状态。
	 * @return -- 服务器的加载情况
	 * @throws Exception
	 */
	public ServerIdxThreadStates queryServerAddState() throws Exception
	{
		ServerIdxThreadStates idxStates =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			idxStates= queryServerAddState(conn);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
	
		return idxStates;
	}
	public ServerIdxThreadStates queryServerAddState(QueryConn conn) throws Exception
	{
		//log.info("Test Server Index Thread State.");
		if (descDb() == null)
			throw new NoDBException();
		
		ServerIdxThreadStates idxStates =new AskQuery(conn).testAddIndex(schema,feedinfo.getCharset());
		if(idxStates==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Test Server Index Thread State Successful .");
		return idxStates;
	}
	/**
	 * 查询库结构
	 * @return -- 返回全文库结构
	 * @throws Exception
	 */
	public Schema descDb() throws Exception
	{
		if (schema == null)
		{
			QueryConn conn=qconn_pool.getConn();
			try
			{
			schema = descDb(conn);
			}catch(Exception e)
			{
				qconn_pool.releaseConn(conn,e);
				throw e;
			}
			qconn_pool.releaseConn(conn);
			
			log.info("Query DB Struction Success.");
		}
		return schema;
	}
	public Schema descDb(BaseConn conn) throws Exception
	{
		if (schema == null)
		{
			schema = new AskQuery(conn).descDb(feedinfo.getCharset());
			if(schema==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
			log.info("Query DB Struction Success.");
		}
		
		return schema;
	}
	/**
	 * 创建一个全文库
	 * @param schema -- 要创建的全文库的结构
	 * @throws Exception
	 */
	public void createDb(Schema schema) throws Exception
	{
		IndexConn conn=iconn_pool.getConn();	
		try
		{
			createDb(conn,schema);
		}catch(Exception e)
		{
			iconn_pool.releaseConn(conn,e);
			throw e;
		}
		iconn_pool.releaseConn(conn);
		
		
	}
	
	public void createDb(IndexConn conn,Schema schema) throws Exception
	{
		log.info("Create DB .");
		if (this.schema != null)
			throw new DBExistException();

		checkNewSchema(schema);
		new AskIndex(conn).createDb(schema,feedinfo.getCharset());
		log.info("Create DB Successful !");
		this.schema = schema;
	}
	

	/**
	 * 检测建库的字段
	 * @param schema
	 * @throws Exception
	 */
	private void checkNewSchema(Schema schema) throws Exception
	{
		int seqnum=schema.howManySequenceField();
		if (seqnum==0)
		{
			Exception ex = new ISearchException("a requence field is required !");
			log.error(ex.getMessage());
			throw ex;
		}

		int sortno = schema.getSortno();
		if (sortno < 0 || sortno >= schema.fieldNum())
			throw new ISearchException("the sortno is error value. ");
		FieldInfo info = schema.get(sortno);
		if (info == null)
			throw new ISearchException("the sortno is error value. ");
		if (!info.isInt32Field() && !info.isInt64Field())
			throw new ISearchException("sort field must int32 field or int64 field. ");
	}


	/**
	 * 加载文章
	 * @param doc -- 要加载的文章
	 * @throws Exception
	 */
	public void addDoc(Doc doc) throws Exception
	{
			IndexConn conn=iconn_pool.getConn();
			try
			{
				addDoc(conn,doc);
			}catch(Exception e)
			{
				iconn_pool.releaseConn(conn,e);
				throw e;
			}
			iconn_pool.releaseConn(conn);
			
			
	}
	public void addDoc(IndexConn conn,Doc doc) throws Exception
	{
	
			log.debug("Remote Add Doc .");
			ByteBuff buf = new ByteBuff();
			ByteBuff tmp = doc.toByteBuff();
			if(tmp==null) return;
			
			if(tmp.getUsed() +4 >FEED_BUf_MAX_LEN)
			{
				log.warn("size of Doc is too big (size="+(tmp.getUsed()+4)+"). please increase index.feed_buffer.size.KB of isearch");
				throw new ISearchException("Remote Add Doc failed !");
			}
			
			buf.append(tmp.getUsed()).append(tmp);

			boolean r=new AskIndex(conn).addDocRemote(buf.array(), buf.getUsed());
			if(r) log.debug("Remote Add Doc Successful !");
			else throw new ISearchException("Remote Add Doc failed !");
		
	}
	

	 public static  int FEED_BUf_MAX_LEN =	1<<20;
	/**
	 * 加载文章
	 * @param docs -- 要加载的文章的集合
	 * @throws Exception
	 */
	public void addDoc(List<? extends Doc> docs) throws Exception
	{
		if(docs==null ||docs.size()==0 )
			return;
			IndexConn conn=iconn_pool.getConn();
			try
			{
				addDoc(conn,docs);
			}catch(Exception e)
			{
				iconn_pool.releaseConn(conn,e);
				throw e;
			}
			iconn_pool.releaseConn(conn);
	}

	public void addDoc(IndexConn conn,List<? extends Doc> docs) throws Exception
	{
		int success=0;
		try
		{
			AskIndex ai=new AskIndex(conn);
			
			log.debug("Remote Add Doc List .");
			ByteBuff buf = new ByteBuff();
			
			
			for (int i = 0; i < docs.size(); i++)
			{
				ByteBuff tmp = docs.get(i).toByteBuff();
				if(tmp==null)
					continue;
				if(tmp.getUsed() +4 >FEED_BUf_MAX_LEN)
				{
					log.warn("size of Doc is too big (size="+(tmp.getUsed()+4)+"). please increase index.feed_buffer.size.KB of isearch");
					continue;
				}
				
				if(buf.getUsed()+tmp.getUsed()+4>FEED_BUf_MAX_LEN)
				{
					boolean r=ai.addDocRemote(buf.array(), buf.getUsed());
					if(r){ log.debug("Remote Add Doc Successful !");
					  success=i;
					}
					else throw new ISearchException("Remote Add Doc failed !");
					
					buf = new ByteBuff();
				}	
				buf.append(tmp.getUsed()).append(tmp);
			}
			boolean r=ai.addDocRemote(buf.array(), buf.getUsed());
			if(r) log.debug("Remote Add Doc Successful !");
			else throw new ISearchException("Remote Add Doc failed !");
		}catch(Exception e)
		{
			log.info("success="+success+" Doc list size="+docs.size());
			for(int i=0;i<success;i++)
			{
				docs.remove(0);
			}	
			throw e;
		}
	}

	
	
	
	public void addDocZiped(List<?extends Doc> docs) throws Exception
	{
		
			IndexConn conn=iconn_pool.getConn();
			try
			{
				addDocZiped(conn,docs);
			}catch(Exception e)
			{
				iconn_pool.releaseConn(conn,e);
				throw e;
			}
			iconn_pool.releaseConn(conn);
		
	}

	public void addDocZiped(IndexConn conn,List<?extends Doc> docs) throws Exception
	{
		if(feedinfo.distributedMode!=2)
			throw new Exception("target is not fearch. cannot sent data to");
		int success=0;
		try
		{
			AskIndex ai=new AskIndex(conn);
			
			log.debug("Remote Add Doc List .");
			ByteBuff buf = new ByteBuff();
			
			
			for (int i = 0; i < docs.size(); i++)
			{
				ByteBuff tmp = docs.get(i).toByteBuff();
				if(tmp==null)
					continue;
				if(tmp.getUsed() +4 >FEED_BUf_MAX_LEN)
				{
					log.warn("size of Doc is too big (size="+(tmp.getUsed()+4)+"). please increase index.feed_buffer.size.KB of isearch");
					continue;
				}
				
				if(buf.getUsed()+tmp.getUsed()+4>FEED_BUf_MAX_LEN)
				{
					boolean r=ai.addDocRemote(buf.array(), buf.getUsed());
					if(r){ log.debug("Remote Add Doc Successful !");
					  success=i;
					}
					else throw new ISearchException("Remote Add Doc failed !");
					
					buf = new ByteBuff();
				}	
				buf.append(tmp.getUsed()).append(tmp);
			}
			boolean r=ai.addZipedDocRemote(buf.array(), buf.getUsed());
			if(r) log.debug("Remote Add Ziped Doc Successful !");
			else throw new ISearchException("Remote Add Ziped Doc failed !");
		}catch(Exception e)
		{
			log.info("success="+success+" Doc list size="+docs.size());
			for(int i=0;i<success;i++)
			{
				docs.remove(0);
			}	
			throw e;
		}
	}
	
	public static String check_table_name(List<String> tables) throws Exception {
		if(tables==null || tables.size()==0)
			throw new Exception("no table input.");
		
		StringBuffer strb =new StringBuffer();
		for(String table_name:tables){
			if(strb.length()>0) strb.append(",");
			check_table_name(table_name);
			strb.append(table_name.toLowerCase());
		}	
		return strb.toString();
	}

	public static void check_table_name(String table_name) throws Exception {
		if(table_name==null || table_name.length()==0)
			throw new Exception("no table name");
		if(table_name.length()>30)
			throw new Exception("the length of  table name >30");
		for(int i=0;i<table_name.length();i++){
			char ch=table_name.charAt(i);
			if(ch>='0' && ch<='9')continue;
			if(ch>='a' && ch<='z')continue;
			if(ch>='A' && ch<='Z')continue;
			if(ch=='_')continue;
			throw new Exception("table name cannot contain this character '"+ch+"'");
		}
	}
	public void addDoc(String table_name,IndexConn conn,List<?extends Doc> docs) throws Exception
	{
		check_table_name(table_name);
		table_name=table_name.toLowerCase();
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
		int success=0;
		try
		{
			AskIndex ai=new AskIndex(conn);
			
			log.debug("Remote Add Doc List .");
			ByteBuff buf = new ByteBuff();
			
			
			for (int i = 0; i < docs.size(); i++)
			{
				ByteBuff tmp = docs.get(i).toByteBuff();
				if(tmp==null)
					continue;
				if(tmp.getUsed() +40 >FEED_BUf_MAX_LEN)
				{
					log.warn("size of Doc is too big (size="+(tmp.getUsed()+4)+"). please increase index.feed_buffer.size.KB of isearch");
					continue;
				}
				
				if(buf.getUsed()+tmp.getUsed()+40>FEED_BUf_MAX_LEN)
				{
					boolean r=ai.addDocRemote(table_name,buf.array(), buf.getUsed());
					if(r){ log.debug("Remote Add Doc Successful !");
					  success=i;
					}
					else throw new ISearchException("Remote Add Doc failed !");
					
					buf = new ByteBuff();
				}	
				buf.append(tmp.getUsed()).append(tmp);
			}
			boolean r=ai.addDocRemote(table_name,buf.array(), buf.getUsed());
			if(r) log.debug("Remote Add Doc Successful !");
			else throw new ISearchException("Remote Add Doc failed !");
		}catch(Exception e)
		{
			log.info("success="+success+" Doc list size="+docs.size());
			for(int i=0;i<success;i++)
			{
				docs.remove(0);
			}	
			throw e;
		}
	}

	public void addDoc(String table_name,List<? extends Doc> docs) throws Exception
	{
		check_table_name(table_name);
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		if(docs==null ||docs.size()==0 )
			return;
		
			IndexConn conn=iconn_pool.getConn();
			try
			{
				addDoc(table_name,conn,docs);
			}catch(Exception e)
			{
				iconn_pool.releaseConn(conn,e);
				throw e;
			}
			iconn_pool.releaseConn(conn);
	}

	public void addDoc(List<String> tables,IndexConn conn,Doc doc) throws Exception
	{
		if(doc==null) return;
		
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");

		if(tables.size()<=Once_Feed_Max_Table){
			addDoc_100t(tables,conn,doc);
			return;
		}
		check_table_name(tables);
		
		List <String>l=new ArrayList<String>();
		for(String t:tables){
			l.add(t);
			if(l.size()==Once_Feed_Max_Table){
				addDoc_100t(l,conn,doc);
				l.clear();
			}
		}
		if(l.size()>0)
			addDoc_100t(l,conn,doc);
	}

	protected void addDoc_100t(List<String> tables,IndexConn conn,Doc doc) throws Exception
	{
		if(doc==null) return;
		
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");

		String tableNames=check_table_name(tables);
		
		System.out.println("add to table:"+tables.toString());
		
		
		
		log.debug("Remote Add Doc .");
		ByteBuff buf = new ByteBuff();
		ByteBuff tmp = doc.toByteBuff();
		if(tmp==null) return;
		
		if(tmp.getUsed() +4 >FEED_BUf_MAX_LEN)
		{
			log.warn("size of Doc is too big (size="+(tmp.getUsed()+4)+"). please increase index.feed_buffer.size.KB of isearch");
			throw new ISearchException("Remote Add Doc failed !");
		}
		
		buf.append(tmp.getUsed()).append(tmp);

		boolean r=new AskIndex(conn).addDocRemote(tableNames,buf.array(), buf.getUsed());
		if(r) log.debug("Remote Add Doc Successful !");
		else throw new ISearchException("Remote Add Doc failed !");
	}
	
	public void addDoc(List<String> tables,Doc doc) throws Exception
	{
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		if(doc==null)
			return;
		
	
			IndexConn conn=iconn_pool.getConn();
			try
			{
				addDoc(tables,conn,doc);
			}catch(Exception e)
			{
				iconn_pool.releaseConn(conn,e);
				throw e;
			}
			iconn_pool.releaseConn(conn);
		
	}

	/**
	 * 通过检索条件进行检索
	 * <p>
	 * 如果只是要统计能够检索到多少条记录。请使用<code>SearchSystem.count(request)</code>
	 * </p>
	 * @param request -- 检索的条件
	 * @return -- 检索结果集
	 * @throws Exception
	 */
	public QueryResponse query(QueryRequest request) throws Exception
	{
		QueryResponse hits =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			hits =query(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		
		return hits;
	}
	
	public QueryResponse query(QueryConn conn,QueryRequest request) throws Exception
	{
		log.info("Query Doc .");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		} 
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		if(request.getQueryType()!=QueryRequest.QUERY_INDEX)
		{
			throw new ISearchException("the query type is not a index query. it is "+request.getQueryTypeName());
		}	
		
		QueryResponse hits =new AskQuery(conn).query(request,this);
		
		if(hits==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Query Doc Successful !");
		return hits;
	}
	
	public DistinctResponse distinct(DistinctRequest request) throws Exception
	{
		DistinctResponse hits =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			hits =distinct(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		return hits;
	}
	public DistinctResponse distinct(QueryConn conn,DistinctRequest request) throws Exception
	{
		log.info("query  Doc distinct .");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		} 
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		if(request.getQueryType()!=QueryRequest.QUERY_DISTINCT_INDEX)
		{
			throw new ISearchException("Please call DistinctRequest.setDisinctBy() first");
		}	
		if(request.getOrderBy()==QueryRequest.OrderBy.random)
			throw new ISearchException("distinct doesnt support random sorter.");
		
		DistinctResponse hits = new AskQuery(conn).distinct_query(request,this);
		
		if(hits==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Query Doc distinct Successful !");
		return hits;
	}
	
	public QueryResponse collectKeyWords(KeyWordRequest request) throws Exception
	{
		QueryResponse hits =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			hits =collectKeyWords(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		return hits;
	}
	public QueryResponse collectKeyWords(QueryConn conn,KeyWordRequest request) throws Exception
	{
		log.info("collectKeyWords  .");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		} 
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		if(request.getQueryType()!=QueryRequest.QUERY_KEYWORD)
		{
			throw new ISearchException("Please call KeyWordRequest.setKeyWordsQueryBy() first");
		}	
		QueryResponse hits =new AskQuery(conn).collectKeyWords(request,this);
		if(hits==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		
		log.info("collectKeyWords Successful !");
		return hits;
	}
	
	/**
	 * 通过条件统计总数
	 * <p>
	 * 只返回统计结果的条数。内容不返回。如果需要检索内容，使用<code>SearchSystem.query(request)</code>
	 * </p>
	 * @param request -- 条件
	 * @return -- 统计的总数
	 * @throws Exception
	 */
	public long count(BaseRequest request) throws Exception
	{
		long relt=0;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			relt =count(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		
		return relt;
	}
	public long count(QueryConn conn,BaseRequest request) throws Exception
	{
		//log.info("Count Doc with not Group.");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		}
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		long relt = new AskQuery(conn).count(request);
		if(relt<0)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Count Doc Successful !");
		return relt;
	}
	/**
	 * 分组统计
	 * @param request -- 检索条件
	 * @return -- 分组信息集合
	 * @throws Exception
	 */
	public GroupResponse group(GroupRequest request) throws Exception
	{
		GroupResponse relt =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			relt =group(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		return relt;
	}
	public GroupResponse group(QueryConn conn,GroupRequest request) throws Exception
	{
		log.info("Count Doc with Group .");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		} 
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		if(request.getQueryType()!=QueryRequest.QUERY_GROUP)
		{
			throw new ISearchException("Please call GroupRequest.setGroupBy() first");
		}
		GroupResponse relt =new AskQuery(conn).group(request,this);
		if(relt==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Count Doc Successful !");
		return relt;
	}

	public CubeResponse cube(CubeRequest request) throws Exception
	{
		CubeResponse relt =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			relt =cube(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		
		return relt;
	}	
	public CubeResponse cube(QueryConn conn,CubeRequest request) throws Exception
	{
		log.info("Count Doc with Cube .");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		} 
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		if(request.getQueryType()!=QueryRequest.QUERY_CUBE)
		{
			throw new ISearchException("Please call CubeRequest.setCubBy() first");
		}
		
		CubeResponse relt =new AskQuery(conn).cube(request,this);
		if(relt==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Count Doc Successful !");
		return relt;
	}	
	/**
	 * 通过条件删除数据
	 * @param request -- 要删除的条件
	 * @return -- 删除的条数
	 * @throws Exception
	 */
	public int delDoc(BaseRequest request) throws Exception
	{
		int v =0;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			v =delDoc(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		
		return v;
	}
	public int delDoc(QueryConn conn,BaseRequest request) throws Exception
	{
		log.info("Remove Doc .");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		}
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		int v =new AskQuery(conn).removeDoc(request);
		if(v<0)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Remove Doc Successful !");
		return v;
	}

	/**
	 * 提交服务器端的内存数据到全文库
	 * <p>
	 * 服务器端有缓存机制，加载的数据会先保存在服务器的缓存的中，直到缓存满了，服务器会自动地提交内存的数据到全文库。调用这个方法，如果服务器缓存中有数据，它会强制提交内存的数据到全文库，而不管缓存是否满了。
	 * </p>
	 * @throws Exception
	 */
	public synchronized void commit() throws Exception
	{
		
		IndexConn conn=iconn_pool.getConn();
		try
		{
			commit(conn);
		}catch(Exception e)
		{
			iconn_pool.releaseConn(conn,e);
			throw e;
		}
		iconn_pool.releaseConn(conn);
	
		
	}
	public synchronized void commit(IndexConn conn) throws Exception
	{
		log.info("DB Commit .");
		
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		boolean r=new AskIndex(conn).commit();
		if(!r)
			log.info("DB Commit failed !");
		else
		    log.info("DB Commit Successful !");
		
	}
	/**
	 * 停止服务器的运行。
	 * @throws Exception
	 */
	public void stopServer() throws Exception
	{
		log.info("Stop server .");
		QueryConn conn=qconn_pool.getConn();
		try
		{
			new AskQuery(conn).stopServer();
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
	

		log.info("Stop Server Successful !");
	}


	

	
	public WordCloudResponse wordcloud(WordCloudRequest request)throws Exception
	{
	
		WordCloudResponse hits =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			hits =wordcloud(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		
		return hits;
	}
	public WordCloudResponse wordcloud(QueryConn conn,WordCloudRequest request)throws Exception
	{
	
		log.info("word cloud  .");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		}
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		if(request.getQueryType()!=QueryRequest.QUERY_WORDCLOUD)
		{
			throw new ISearchException("Please call WordCloudRequest.setWordCloudBy() first");
		}
		
		WordCloudResponse hits =new AskQuery(conn).wordCould(request,this);
		if(hits==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Query Doc Successful !");
		return hits;
	}

	public WamResponse wam(WamRequest request) throws Exception
	{
		WamResponse hits =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			hits =wam(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		
		return hits;
	}
	public WamResponse wam(QueryConn conn,WamRequest request) throws Exception
	{
		log.info("WAM  .");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		}
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		if(request.getQueryType()!=QueryRequest.QUERY_WAM)
		{
			throw new ISearchException("Please call WamRequest.setWAMBy() first");
		}
		WamResponse hits =new AskQuery(conn).wordAssoociateMap(request,this);
		if(hits==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("WAM Successful !");
		return hits;
	}	
	public ClusterResponse cluster(ClusterRequest request) throws Exception{
		ClusterResponse hits =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			hits =cluster(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
	
		return hits;
		
	}
	public ClusterResponse cluster(QueryConn conn,ClusterRequest request) throws Exception{
		log.info("cluster  .");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		}
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		if(request.getQueryType()!=QueryRequest.QUERY_CLUSTER)
		{
			throw new ISearchException("Please call ClusterRequest.setClusterBy() first");
		}
		
		if(feedinfo.version % 1000<128)
			request.getCritHeader().clusterWordTotalMaxLimit=3;
		

		ClusterResponse hits = new AskQuery(conn).cluster(request,this); 
		if(hits==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("cluster Successful !");
		return hits;
		
	}	
	public void emptyServer()  throws Exception{
		

		IndexConn conn=iconn_pool.getConn();
		try
		{
			emptyServer(conn);
		}
		catch(Exception e)
		{
			iconn_pool.releaseConn(conn,e);
			throw e;
		}
		iconn_pool.releaseConn(conn);
		
		
	}
	public void emptyServer(IndexConn conn)  throws Exception{
		log.info("empty server .");
		new AskIndex(conn).empty();
		log.info("empty Server Successful !");
	}
   
    
   
   
  
    
    public List<XWord> tokenText(String text)throws Exception
    {
    	List<XWord> lw =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			lw =tokenText(conn,text);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		return lw;
    }
    public List<XWord> tokenText(QueryConn conn,String text)throws Exception
    {
		List<XWord> lw = new AskMining(conn).tokenText(text,getCharset()); 
		return lw;
    }
    public String fingerPrint(String text)throws Exception
    {
    	String fp=null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			fp =fingerPrint(conn,text);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		return fp;
    }
    public String fingerPrint(QueryConn conn,String text)throws Exception
    {
    	String fp = new AskMining(conn).fingerPrint(TxUtil.visibleChars(text),getCharset()); 
		return fp;
    }
    
    public void destroy()
    {
    	qconn_pool.destroy();
    	iconn_pool.destroy();
    	log.info("search api destory");
    }
    public IndexConn createIndexConn() throws Exception {
		
		return iconn_pool.getConn();
	}
	public void releaseIndexConn(IndexConn conn) throws Exception {
		iconn_pool.releaseConn(conn);
		
	}
	public QueryConn createQueryConn() throws Exception {
		 return qconn_pool.getConn();
	}
	public void releaseQueryConn(QueryConn conn) throws Exception {
		qconn_pool.releaseConn(conn);
		
	}

	public ConnPool<QueryConn> getQueryConnPool() {
		return qconn_pool;
	}
	public ConnPool<IndexConn> getIndexConnPool() {
		return iconn_pool;
	}
	
	public UpdateResponse update(QueryRequest req,Doc doc) throws Exception
	{
		UpdateResponse hits ;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			hits =update(conn,req,doc);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		
		return hits;
	}
	public UpdateResponse update(QueryConn conn,BaseRequest request,Doc doc) throws Exception
	{
		log.info(" Doc Update.");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		}
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		doc.verify4update();
		
			
		
		if(request.getQueryType()!=QueryRequest.QUERY_INDEX)
		{
			throw new ISearchException("the query type is not a common query");
		}	
		while(true)
		{	
			UpdateResponse r =new AskQuery(conn).update(request,doc,this);
		
			if(r==null)
			{	if(conn.getErrCode()!=26)
					throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
				Thread.sleep(1000);
				log.info("Update Doc meet lock error, try again.");
			}
			else
			{	log.info("Update Doc Successful! Changed number is "+r);
				return r;
			}	
		}	
	}
	
	
	public UpdateResponse update(long seq_value_of_doc,Doc doc) throws Exception
	{
		QueryRequest req = new QueryRequest(this);
		req.createCriteria().andIn(schema.getSeqFI().getName(), new long[]{seq_value_of_doc});
		return update(req,doc);
	}
	public UpdateResponse update(QueryConn conn,long seq_value_of_doc,Doc doc) throws Exception
	{
		BaseRequest req = new BaseRequest(this);
		req.createCriteria().andIn(schema.getSeqFI().getName(), new long[]{seq_value_of_doc});
		return update(conn,req,doc);
	}
	
	
	public byte[] fingerPrintSimhash(String text)throws Exception
    {
    	byte[] fp=null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			fp =fingerPrintSimhash(conn,text);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		return fp;
    }
    public byte[] fingerPrintSimhash(QueryConn conn,String text)throws Exception
    {
    	if(feedinfo.version%1000<105)
    		throw new Exception("version of isearch is low, this function is not supported.");
    	return new AskMining(conn).fingerPrintSimhash(TxUtil.visibleChars(text),getCharset()); 
    }
    
    /**
     * maxline 返回数据最多句子数
     * sentence_end_with_period 句子必须以句号结束，除了标题
     * */
    public String summary(String text,int maxline,boolean sentence_end_with_period)throws Exception
    {
    	String fp=null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			fp =summary(conn,text,maxline,sentence_end_with_period);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		return fp;
    }
    public String summary(QueryConn conn,String text,int maxline,boolean sentence_end_with_period)throws Exception
    {
    	if(feedinfo.version%1000<105)
    		throw new Exception("version of isearch is low, this function is not supported.");
    	return new AskMining(conn).summary(text,getCharset(),maxline,sentence_end_with_period); 
    }

    public SummaryResponse summary_query(QueryRequest request)throws Exception
    {
    	SummaryResponse fp=null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			fp =summary_query(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		return fp;
    }
    public SummaryResponse summary_query(QueryConn conn,QueryRequest request)throws Exception
    {
    	if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		}
    	if(feedinfo.version%1000<126)
    		throw new Exception("version of isearch is low, this function is not supported.");
    	if(request.getCritHeader().summaryQueryUndefined())
    		throw new ISearchException("Please call QueryRequest.setSummaryQuery() first");
    	return new AskMining(conn).summary_query(request,this); 
    }
    
    public DbSeqScope querySeqScope() throws Exception
	{
    	if(feedinfo.version%1000<132)
    		throw new Exception("version of isearch is low, this function is not supported.");
    	
    	DbSeqScope scope =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			scope =querySeqScope(conn);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		
		return scope;
	}
	
	public DbSeqScope querySeqScope(QueryConn conn) throws Exception
	{
		if(feedinfo.version%1000<132)
    		throw new Exception("version of isearch is low, this function is not supported.");
		
		log.info("Query Seq scope .");
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		
		
		DbSeqScope scope =new AskQuery(conn).querySeqScope();
		
		if(scope==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Query seq scope Successful !");
		return scope;
	}
	
	
	public UnitedResponse unitedQuery(QueryConn conn,UnitedRequest request) throws Exception
	{
		log.info("unitedQuery Doc .");
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		}
		
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
//		if(request.getQueryType()!=QueryRequest.UNITED_QUERY)
//		{
//			throw new ISearchException("the query type is not a unitedQuery. it is "+request.getQueryTypeName());
//		}	
		request.verify();
		
		UnitedResponse hits =new AskQuery(conn).unitedQuery(request,this);
		
		if(hits==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Query Doc Successful !");
		return hits;
	}
	
	public UnitedResponse unitedQuery(UnitedRequest request) throws Exception
	{
		UnitedResponse hits =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			hits =unitedQuery(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		return hits;
	}

	
	public ExamineResponse examine(QueryRequest request) throws Exception
	{
		ExamineResponse hits =null;
		QueryConn conn=qconn_pool.getConn();
		try
		{
			hits =examine(conn,request);
		}catch(Exception e)
		{
			qconn_pool.releaseConn(conn,e);
			throw e;
		}
		qconn_pool.releaseConn(conn);
		
		
		return hits;
	}
	
	public ExamineResponse examine(QueryConn conn,QueryRequest request) throws Exception
	{
		log.info("Examine .");
		
		if(feedinfo.distributedMode== 3 && StringUtil.isEmpty(request.getHeader().getTableName())){
			throw new Exception("please input table name.");	
		} 
		
		if (null == this.schema)
		{
			Exception ex = new NoDBException();
			log.error(ex.getMessage());
			throw ex;
		}
		
		ExamineResponse hits =new AskQuery(conn).examine(request,this);
		
		if(hits==null)throw new ISearchException(conn.getErrCode(),conn.getErrMsg());
		log.info("Query Doc Successful !");
		return hits;
	}
	public boolean isSupportTable() {
		return feedinfo.distributedMode==3;
	}	
	
	public boolean createTable(String table_name,IndexConn conn) throws Exception
	{
		check_table_name(table_name);
		table_name=table_name.toLowerCase();
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			AskIndex ai=new AskIndex(conn);
			
			boolean r=ai.createTable(table_name);
			if(r) log.debug("Create table["+table_name+"] Successful !");
			else throw new ISearchException("Create table["+table_name+"] failed !");
			return r;
	}

	public boolean createTable(String table_name) throws Exception
	{
		boolean r=false;
		check_table_name(table_name);
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			IndexConn conn=iconn_pool.getConn();
			try
			{
				r=createTable(table_name,conn);
			}catch(Exception e)
			{
				iconn_pool.releaseConn(conn,e);
				throw e;
			}
			iconn_pool.releaseConn(conn);
			return r;
	}
	public boolean dropTable(String table_name,IndexConn conn) throws Exception
	{
		check_table_name(table_name);
		table_name=table_name.toLowerCase();
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			AskIndex ai=new AskIndex(conn);
			
			boolean r=ai.dropTable(table_name);
			if(r) log.debug("drop table["+table_name+"] Successful !");
			else throw new ISearchException("drop table["+table_name+"] failed !");
			return r;
	}

	public boolean dropTable(String table_name) throws Exception
	{
		boolean r=false;
		check_table_name(table_name);
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			IndexConn conn=iconn_pool.getConn();
			try
			{
				r=dropTable(table_name,conn);
			}catch(Exception e)
			{
				iconn_pool.releaseConn(conn,e);
				throw e;
			}
			iconn_pool.releaseConn(conn);
			return r;
	}
	public boolean accessTable(String table_name,QueryConn conn) throws Exception
	{
		check_table_name(table_name);
		table_name=table_name.toLowerCase();
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			AskQuery ai=new AskQuery(conn);
			
			boolean r=ai.accessTable(table_name);
			return r;
	}
	/*
	 * 访问table 返回true代表表存在，false代表表不存在
	 * */
	public boolean accessTable(String table_name) throws Exception
	{
		boolean r=false;
		check_table_name(table_name);
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			QueryConn conn=qconn_pool.getConn();
			try
			{
				r=accessTable(table_name,conn);
			}catch(Exception e)
			{
				qconn_pool.releaseConn(conn,e);
				throw e;
			}
			qconn_pool.releaseConn(conn);
			return r;
	}		
	
	public void emptyTable(String table_name)  throws Exception{
		

		IndexConn conn=iconn_pool.getConn();
		try
		{
			emptyTable(conn,table_name);
		}
		catch(Exception e)
		{
			iconn_pool.releaseConn(conn,e);
			throw e;
		}
		iconn_pool.releaseConn(conn);
		
		
	}
	public void emptyTable(IndexConn conn,String table_name)  throws Exception{
		check_table_name(table_name);
		table_name=table_name.toLowerCase();
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
		log.info("empty table("+table_name+") .");
		new AskIndex(conn).emptyTable(table_name);
		log.info("empty table("+table_name+") Successful !");
	}
   
	
	public boolean loadTable(String table_name,IndexConn conn) throws Exception
	{
		check_table_name(table_name);
		table_name=table_name.toLowerCase();
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			AskIndex ai=new AskIndex(conn);
			
			boolean r=ai.loadTable(table_name);
			if(r) log.debug("Load table["+table_name+"] Successful !");
			else throw new ISearchException("Load table["+table_name+"] failed !");
			return r;
	}

	public boolean loadTable(String table_name) throws Exception
	{
		boolean r=false;
		check_table_name(table_name);
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			IndexConn conn=iconn_pool.getConn();
			try
			{
				r=loadTable(table_name,conn);
			}catch(Exception e)
			{
				iconn_pool.releaseConn(conn,e);
				throw e;
			}
			iconn_pool.releaseConn(conn);
			return r;
	}
	public boolean unloadTable(String table_name,IndexConn conn) throws Exception
	{
		check_table_name(table_name);
		table_name=table_name.toLowerCase();
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			AskIndex ai=new AskIndex(conn);
			
			boolean r=ai.unloadTable(table_name);
			if(r) log.debug("unload table["+table_name+"] Successful !");
			else throw new ISearchException("unload table["+table_name+"] failed !");
			return r;
	}

	public boolean unloadTable(String table_name) throws Exception
	{
		boolean r=false;
		check_table_name(table_name);
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			IndexConn conn=iconn_pool.getConn();
			try
			{
				r=unloadTable(table_name,conn);
			}catch(Exception e)
			{
				iconn_pool.releaseConn(conn,e);
				throw e;
			}
			iconn_pool.releaseConn(conn);
			return r;
	}
	
	public List<String> readTableNames(QueryConn conn) throws Exception
	{
		
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			AskQuery ai=new AskQuery(conn);
			
			List<String> r=ai.readTableNames();
			return r;
	}
	/*
	 * 访问table 返回true代表表存在，false代表表不存在
	 * */
	public List<String> readTableNames() throws Exception
	{
		List<String> r=null;
		
		if(feedinfo.distributedMode!=3)
			throw new Exception("unsupport table.");
		
			QueryConn conn=qconn_pool.getConn();
			try
			{
				r=readTableNames(conn);
			}catch(Exception e)
			{
				qconn_pool.releaseConn(conn,e);
				throw e;
			}
			qconn_pool.releaseConn(conn);
			return r;
	}		
	public AsynInfo asynQueryStatus(QueryConn conn) throws Exception
	{
		
		if(feedinfo.distributedMode==3)
			throw new Exception("unsupported.");
		
			AskQuery ai=new AskQuery(conn);
			
			AsynInfo r=ai.asynQueryStatus();
			return r;
	}

	public AsynInfo asynQueryStatus() throws Exception
	{
		
		if(feedinfo.distributedMode==3)
			throw new Exception("unsupported.");
		
		AsynInfo r=null;
		
		
			QueryConn conn=qconn_pool.getConn();
			try
			{
				r=asynQueryStatus(conn);
			}catch(Exception e)
			{
				qconn_pool.releaseConn(conn,e);
				throw e;
			}
			qconn_pool.releaseConn(conn);
			return r;
	}		
	public AsynInfo asynQueryBegin(QueryConn conn,UnitedRequest req) throws Exception
	{
		if(feedinfo.distributedMode==3)
			throw new Exception("unsupported.");
	
		
			AskQuery ai=new AskQuery(conn);
			
			AsynInfo r=ai.asynQueryBegin(0,req);
			return r;
	}

	public AsynInfo asynQueryBegin(UnitedRequest req) throws Exception
	{
		
		if(feedinfo.distributedMode==3)
			throw new Exception("unsupported.");
		AsynInfo r=null;
		
		
			QueryConn conn=qconn_pool.getConn();
			try
			{
				r=asynQueryBegin(conn,req);
			}catch(Exception e)
			{
				qconn_pool.releaseConn(conn,e);
				throw e;
			}
			qconn_pool.releaseConn(conn);
			return r;
	}		
	public AsynInfo asynQueryResult(QueryConn conn,int id) throws Exception
	{
		
		if(feedinfo.distributedMode==3)
			throw new Exception("unsupported.");
		
			AskQuery ai=new AskQuery(conn);
			
			AsynInfo r=ai.asynQueryResult(id,this);
			return r;
	}

	public AsynInfo asynQueryResult(int id) throws Exception
	{
		
		if(feedinfo.distributedMode==3)
			throw new Exception("unsupported.");
		
		AsynInfo r=null;
		
		
			QueryConn conn=qconn_pool.getConn();
			try
			{
				r=asynQueryResult(conn,id);
			}catch(Exception e)
			{
				qconn_pool.releaseConn(conn,e);
				throw e;
			}
			qconn_pool.releaseConn(conn);
			return r;
	}		
	public AsynInfo asynQueryStop(QueryConn conn,int id) throws Exception
	{
		
		if(feedinfo.distributedMode==3)
			throw new Exception("unsupported.");
		
			AskQuery ai=new AskQuery(conn);
			
			AsynInfo r=ai.asynQueryStop(id);
			return r;
	}

	public AsynInfo asynQueryStop(int id) throws Exception
	{
		
		if(feedinfo.distributedMode==3)
			throw new Exception("unsupported.");
		
		AsynInfo r=null;
		
		
			QueryConn conn=qconn_pool.getConn();
			try
			{
				r=asynQueryStop(conn,id);
			}catch(Exception e)
			{
				qconn_pool.releaseConn(conn,e);
				throw e;
			}
			qconn_pool.releaseConn(conn);
			return r;
	}
		
}
