package cn.someget.cache.service.impl;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import cn.someget.cache.service.CacheService;
import cn.someget.cache.utils.LocalCache;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 本地缓存相应的处理
 *
 * @author zyf
 * @date 2022-05-03 16:05
 */
@Service("defaultLocalCacheService")
public class LocalCacheServiceImpl implements CacheService {

    @Resource(name = "defaultRedisCacheService")
    private CacheService defaultRedisCacheService;
    @Resource
    private LocalCache localCache;

    @Override
    public <V> V getObjectFromCache(String key, Class<V> clazz) {
        if (StrUtil.isBlank(key)) {
            return null;
        }
        V value = (V) localCache.getIfPresent(key);
        if (value != null) {
            return value;
        }

        V v = defaultRedisCacheService.getObjectFromCache(key, clazz);
        if (v == null) {
            return null;
        }
        localCache.put(key, v);
        return v;
    }

    @Override
    public <V> List<V> getObjectListFromCache(String key, Class<V> clazz) {
        if (StrUtil.isBlank(key)) {
            return new ArrayList<>();
        }
        List<V> value = (List<V> ) localCache.getIfPresent(key);
        if (value != null) {
            return value;
        }

        List<V> v = defaultRedisCacheService.getObjectListFromCache(key, clazz);
        if (CollectionUtils.isEmpty(v)) {
            return new ArrayList<>();
        }
        localCache.put(key, v);
        return v;
    }

    @Override
    public <K, V> Map<K, V> getObjectFromCache(List<K> ids, Class<V> clazz, String prefix) {
        if (CollectionUtils.isEmpty(ids)) {
            return new HashMap<>(0);
        }
        ids.removeIf(Objects::isNull);
        List<String> keys = ids.stream().map(p -> String.format(prefix, p)).collect(Collectors.toList());
        Map<String, V> allPresent = (Map<String, V>) localCache.getAllPresent(keys);
        Map<K, V> results = new HashMap<>(ids.size());
        List<K> missIds = new ArrayList<>();
        for (K id : ids) {
            String key = String.format(prefix, id);
            V v = allPresent.get(key);
            if (v != null) {
                results.put(id, v);
            } else {
                missIds.add(id);
            }
        }

        if (CollectionUtils.isEmpty(missIds)) {
            return results;
        }

        Map<K, V> objectFromCache = defaultRedisCacheService.getObjectFromCache(missIds, clazz, prefix);
        Map<String, Object> needCacheObjects = new HashMap<>(missIds.size());
        for (K k : missIds) {
            V v = objectFromCache.get(k);
            if (v == null) {
                continue;
            }
            needCacheObjects.put(String.format(prefix, k), v);
            results.put(k, v);
        }
        if (MapUtil.isNotEmpty(needCacheObjects)) {
            localCache.putAll(needCacheObjects);
        }
        return results;
    }

    @Override
    public <K, V> Map<K, List<V>> getObjectListFromCache(List<K> ids, Class<V> clazz, String prefix) {
        if (CollectionUtils.isEmpty(ids)) {
            return new HashMap<>(0);
        }
        ids.removeIf(Objects::isNull);
        List<String> keys = ids.stream().map(p -> String.format(prefix, p)).collect(Collectors.toList());
        Map<String, Object> allPresent = localCache.getAllPresent(keys);
        Map<K, List<V>> results = new HashMap<>(ids.size());
        List<K> missIds = new ArrayList<>();
        for (K id : ids) {
            String key = String.format(prefix, id);
            List<V> v = (List<V>) allPresent.get(key);
            if (v != null) {
                results.put(id, v);
            } else {
                missIds.add(id);
            }
        }

        if (CollectionUtils.isEmpty(missIds)) {
            return results;
        }

        Map<K, List<V>> objectListFromCache = defaultRedisCacheService.getObjectListFromCache(missIds, clazz, prefix);
        Map<String, Object> needCacheObjects = new HashMap<>(missIds.size());
        for (K k : missIds) {
            List<V> v = objectListFromCache.get(k);
            if (v == null) {
                continue;
            }
            needCacheObjects.put(String.format(prefix, k), v);
            results.put(k, v);
        }
        if (MapUtil.isNotEmpty(needCacheObjects)) {
            localCache.putAll(needCacheObjects);
        }
        return results;
    }

}
