/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.sliew.carp.framework.queue.kekio.message;

import com.fasterxml.jackson.annotation.JsonTypeInfo;
import lombok.Getter;

import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

@JsonTypeInfo(use = JsonTypeInfo.Id.NAME, property = Message.JSON_NAME_PROPERTY)
public abstract class Message implements Serializable {

    public static final String JSON_NAME_PROPERTY = "kind";
    public static final Long DEFAULT_ACK_TIMEOUT_MS = Duration.ofSeconds(5L).toMillis();

    @Getter
    private final List<Attribute> attributes = new ArrayList<>();

    @SuppressWarnings("unchecked")
    public <A extends Attribute> A getAttribute(Class<A> attributeClass) {
        return (A) attributes.stream()
                .filter(attributeClass::isInstance)
                .findFirst()
                .orElse(null);
    }

    public <A extends Attribute> A setAttribute(A attribute) {
        attributes.removeIf(attr -> attr.getClass().equals(attribute.getClass()));
        attributes.add(attribute);
        return attribute;
    }

    /**
     * Defines an ack timeout override in milliseconds, a null value will use the
     * queue default.
     */
    public Long getAckTimeoutMs() {
        return DEFAULT_ACK_TIMEOUT_MS;
    }
}
