/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.sliew.carp.framework.pf4j.core.update.downloader;

import org.pf4j.update.FileDownloader;
import org.pf4j.update.SimpleFileDownloader;

import java.io.IOException;
import java.net.URL;
import java.nio.file.Path;
import java.util.List;

/**
 * Composes {@link SupportingFileDownloader}s, downloading through whichever supports the given file URL. If no provided
 * {@code fileDownloaders} match a given file URL, {@link SimpleFileDownloader} is used as a fallback.
 */
public class CompositeFileDownloader implements FileDownloader {

    private final List<SupportingFileDownloader> fileDownloaders;

    public CompositeFileDownloader(List<SupportingFileDownloader> fileDownloaders) {
        this.fileDownloaders = fileDownloaders;
    }

    @Override
    public Path downloadFile(URL fileUrl) throws IOException {
        return fileDownloaders.stream()
                .filter(downloader -> downloader.supports(fileUrl))
                .findFirst()
                .map(downloader -> {
                    try {
                        return downloader.downloadFile(fileUrl);
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                })
                .orElseGet(() -> {
                    try {
                        return new SimpleFileDownloader().downloadFile(fileUrl);
                    } catch (Exception e) {
                        throw new RuntimeException(e);
                    }
                });
    }
}
