/*
 * Copyright (C) 2020-2025, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.license.model;

import cn.sinozg.applet.common.utils.CypherUtil;
import cn.sinozg.applet.license.LicenseUtil;
import cn.sinozg.applet.license.core.CustomLicenseManager;
import de.schlichtherle.license.LicenseContent;
import de.schlichtherle.license.LicenseManager;
import de.schlichtherle.util.ObfuscatedString;
import java.io.File;
import javax.security.auth.x500.X500Principal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 证书生成
 * @Description
 * @Copyright Copyright (c) 2025
 * @author xyb
 * @since 2025-03-23 17:38:51
*/
public class LicenseCreator {

    private static final Logger log = LoggerFactory.getLogger(LicenseCreator.class);

    private final static X500Principal DEFAULT_PRINCIPAL = new X500Principal("CN=sinozg.cn, OU=sinozg, O=sinozg, L=WH, ST=HB, C=CN");

    private static final String S = new ObfuscatedString(new long[] {0x47BCC194C3673035L, 0x3CE76171EC5420A5L, 0xBAB4D6E4AE375EFBL, 0xC640C81979E73F13L, 0xF05987A27F98A43BL, 0x484DA33759AEA94EL, 0x630E57053DE48660L, 0xBFD037A30E9E2C7L, 0xDA1EA76758D65E5FL, 0xAD3AC6D7932672C0L, 0x91C5FB606FD0B285L, 0xBE6D95CF36425DCL, 0xB6DF8EA69E2EE4B5L, 0x9077B0EE342F810DL, 0xE528DBE5059FEDL, 0x41BC1C039DAB0104L, 0x90299BB65809CBE9L, 0x704E6882F1B4E8D5L, 0xC785A993006895F2L, 0xA9840E989DEA4F13L, 0x85052ED9B89B90A8L, 0x17B201DBE671886DL, 0xA38C31138909B66CL, 0x8A3689FB13621E2DL, 0x147F65C37B5CD840L, 0xA6254DBAADE1F7FFL, 0xEC9710565CD7D98L, 0x626A0D560201DC91L}).toString();

    public boolean createLicense(LicenseCreateParams params) {
        try {
            LicenseManager manager = new CustomLicenseManager(LicenseCreator.class, params);
            LicenseContent content = licenseContent(params);
            manager.store(content, new File(params.getPath()));
            return true;
        } catch (Exception e) {
            log.error("证书生成失败：{}", LicenseUtil.errorMessage(e));
            return false;
        }
    }

    private LicenseContent licenseContent(LicenseCreateParams params){
        LicenseContent content = new LicenseContent();
        content.setHolder(DEFAULT_PRINCIPAL);
        content.setIssuer(DEFAULT_PRINCIPAL);
        content.setSubject(params.getSubject());
        content.setIssued(LicenseUtil.toDate(params.getIssued()));
        content.setNotBefore(LicenseUtil.toDate(params.getIssued()));
        content.setNotAfter(LicenseUtil.toDate(params.getExpiry()));
        content.setConsumerType(params.getType());
        content.setConsumerAmount(params.getAmount());
        content.setInfo(params.getDesc());
        // 扩展校验服务器硬件信息
        LicenseCheckModel model = params.getCheckModel();
        if (model == null) {
            model = new LicenseCheckModel();
        }
        String sign = CypherUtil.signMessage(true, S, model.getCpuSerial(), model.getMainBoardSerial(),
                LicenseUtil.join(model.getIpAddress()), LicenseUtil.join(model.getMacAddress()));
        model.setSign(sign);
        content.setExtra(model);
        return content;
    }
}
