package cn.sinozg.applet.common.suport;

import org.apache.commons.fileupload2.core.DiskFileItem;
import org.jetbrains.annotations.NotNull;
import org.springframework.util.FileCopyUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

/**
 * Spring MultipartFile adapter, wrapping a Servlet Part object.
 * @author xyb
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-07-15 22:01
 */
public class StandardMultipartFile implements MultipartFile, Serializable  {

    private final DiskFileItem fileItem;

    private final String filename;

    public StandardMultipartFile(DiskFileItem fileItem, String filename) {
        this.fileItem = fileItem;
        this.filename = filename;
    }

    @Override
    public @NotNull String getName() {
        return this.fileItem.getName();
    }

    @Override
    public String getOriginalFilename() {
        return this.filename;
    }

    @Override
    public String getContentType() {
        return this.fileItem.getContentType();
    }

    @Override
    public boolean isEmpty() {
        return (this.fileItem.getSize() == 0);
    }

    @Override
    public long getSize() {
        return this.fileItem.getSize();
    }

    @Override
    public byte @NotNull [] getBytes() throws IOException {
        return FileCopyUtils.copyToByteArray(this.fileItem.getInputStream());
    }

    @Override
    public @NotNull InputStream getInputStream() throws IOException {
        return this.fileItem.getInputStream();
    }

    @Override
    public void transferTo(File dest) throws IOException, IllegalStateException {
        this.fileItem.write(Paths.get(dest.getPath()));
        if (dest.isAbsolute() && !dest.exists()) {
            // Servlet Part.write is not guaranteed to support absolute file paths:
            // may translate the given path to a relative location within a temp dir
            // (e.g. on Jetty whereas Tomcat and Undertow detect absolute paths).
            // At least we offloaded the file from memory storage; it'll get deleted
            // from the temp dir eventually in any case. And for our user's purposes,
            // we can manually copy it to the requested location as a fallback.
            FileCopyUtils.copy(this.fileItem.getInputStream(), Files.newOutputStream(dest.toPath()));
        }
    }

    @Override
    public void transferTo(@NotNull Path dest) throws IOException, IllegalStateException {
        FileCopyUtils.copy(this.fileItem.getInputStream(), Files.newOutputStream(dest));
    }
}
