package cn.sinozg.applet.common.aspectj;

import cn.sinozg.applet.common.annotation.RedissonLock;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.constant.BaseRedisKeys;
import cn.sinozg.applet.common.enums.LockGranularityType;
import cn.sinozg.applet.common.utils.DateUtil;
import cn.sinozg.applet.common.utils.OsUtil;
import cn.sinozg.applet.common.utils.PojoWebUtil;
import cn.sinozg.applet.common.utils.RedisUtil;
import javax.annotation.Resource;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.concurrent.TimeUnit;

/**
 * 分布式锁实现
 * 若注解用于service类，让该切面优先于Transactional注解
 *
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2023
 * @since 2023-08-24 15:29:56
 */
@Order(1)
@Aspect
@Component
public class RedissonLockAspect {

    @Resource
    private RedissonClient redissonClient;

    private final Logger log = LoggerFactory.getLogger(RedissonLockAspect.class);

    /**
     * 配置织入点
     */
    @Pointcut("@annotation(cn.sinozg.applet.common.annotation.RedissonLock)")
    public void lockPointCut() {
    }

    @Around("lockPointCut()")
    public Object around(ProceedingJoinPoint jp) throws Throwable {
        Object obj = null;
        // 获得注解
        RedissonLock annotation = PojoWebUtil.getAnnotation(jp, RedissonLock.class);
        // 拿到key 包类名称_方法名称_key
        // 设置方法名称
        String name = PojoWebUtil.methodKey(jp, BaseConstants.UNDERLINE);
        String key = annotation.key();
        if (StringUtils.isBlank(key)) {
            key = BaseConstants.AT;
        } else {
            key = PojoWebUtil.getElValue(jp, key);
        }
        // 如果是定时任务
        boolean execute = scheduled(annotation, name);
        if (execute) {
            // 获取到锁
            String redisKey = String.format(BaseRedisKeys.REDISSON_LOCK, name, key);
            RLock lock = null;
            try {
                lock = redissonClient.getLock(redisKey);
                boolean hasLock = false;
                if (lock != null) {
                    hasLock = lock.tryLock(annotation.waitTime(), annotation.leaseTime(), TimeUnit.SECONDS);
                }
                String ip = OsUtil.getHostIp();
                if (hasLock) {
                    log.info("{}获取到锁，{}，执行后续任务", ip, redisKey);
                    obj = jp.proceed();
                } else {
                    log.error("{}没有获取到锁，{}", ip, redisKey);
                }
            } finally {
                if (lock != null && lock.isLocked()) {
                    if (lock.isHeldByCurrentThread()) {
                        lock.unlock();
                    }
                }
            }
        }
        return obj;
    }

    /**
     * 如果时定时任务 判断周期内是否已经执行
     * @param lock 注解
     * @param name 方法名
     * @return 是否执行
     */
    private boolean scheduled(RedissonLock lock, String name){
        if (!lock.scheduled()) {
            return true;
        }
        LockGranularityType type = lock.granularity();
        String time;
        if (type == LockGranularityType.DAY) {
            time = DateUtil.formatDate(LocalDate.now(), type.getPattern());
        } else {
            time = DateUtil.formatDateTime(LocalDateTime.now(), type.getPattern());
        }
        String flagKey = String.format(BaseRedisKeys.REDISSON_LOCK_FLAG, name, time);
        String flag = RedisUtil.getCacheObject(flagKey);
        boolean getLock = StringUtils.isBlank(flag);
        // 设置 flag key 设置1天的时间
        if (getLock) {
            RedisUtil.setCacheObject(flagKey, BaseConstants.IGNORE, Duration.ofDays(1));
        }
        return getLock;
    }
}



