/**
 * Copyright (c) 2019, Sinlmao (888@1st.com).
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.sinlmao.commons.network.utils;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * <p> program: Sinlmao Commons Network Utils
 * <p> description: Jackson工具类
 * <p> create: 2021-03-17 23:47
 *
 * @author Sinlmao
 * @since v1.6.0
 */
public class JacksonUtil {

    private final static ObjectMapper objectMapper = new ObjectMapper();

    private JacksonUtil() {

    }

    public static ObjectMapper getInstance() {
        return objectMapper;
    }

    /**
     * JavaBean、列表数组转换为JSON字符串
     *
     * @param obj 对象
     * @return JSON字符串
     * @throws Exception 异常抛出
     */
    public static String toJSONString(Object obj) throws JsonProcessingException {
        return objectMapper.writeValueAsString(obj);
    }

    /**
     * JavaBean、列表数组转换为JSON字符串，忽略空值
     *
     * @param obj 对象
     * @return JSON字符串
     * @throws Exception 异常抛出
     */
    public static String toJSONStringIgnoreNull(Object obj) throws JsonProcessingException {
        ObjectMapper mapper = new ObjectMapper();
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        return mapper.writeValueAsString(obj);
    }

    /**
     * JSON字符串转JavaBean
     *
     * @param json  JSON字符串
     * @param clazz 目标对象类型
     * @param <T>   泛型
     * @return 目标对象
     * @throws Exception 异常抛出
     */
    public static <T> T toJavaObject(String json, Class<T> clazz) throws JsonProcessingException {
        ObjectMapper mapper = new ObjectMapper();
        mapper.configure(DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY, true);
        return mapper.readValue(json, clazz);
    }

    /**
     * JSON字符串转换为Map
     *
     * @param json JSON字符串
     * @return Map对象
     * @throws Exception 异常抛出
     */
    @SuppressWarnings("rawtypes")
    public static Map toMap(String json) throws JsonProcessingException {
        ObjectMapper mapper = new ObjectMapper();
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        return mapper.readValue(json, Map.class);
    }

    /**
     * JSON字符串转换为指定值类型的Map
     *
     * @param json  JSON字符串
     * @param clazz 目标对象类型
     * @param <T>   泛型
     * @return Map对象
     * @throws Exception 异常抛出
     */
    public static <T> Map<String, T> toMap(String json, Class<T> clazz) throws JsonProcessingException {
        Map<String, Map<String, T>> map = objectMapper.readValue(json,
                new TypeReference<Map<String, Map<String, T>>>() {
                });
        Map<String, T> result = new HashMap<>();
        for (Map.Entry<String, Map<String, T>> entry : map.entrySet()) {
            result.put(entry.getKey(), mapToBean(entry.getValue(), clazz));
        }
        return result;
    }

    /**
     * JSON字符串深度转换为Map，如果Map内部的元素存在JSON字符串，继续解析
     *
     * @param json JSON字符串
     * @return Map对象
     * @throws Exception 异常抛出
     */
    public static Map<String, Object> toMapNeedDeeply(String json) throws JsonProcessingException {
        return toMapRecursion(json, objectMapper);
    }

    /**
     * JSON字符串转换为List，如果List内部的元素存在JSON字符串，继续解析
     *
     * @param json   JSON字符串
     * @param mapper ObjectMapper
     * @return List对象
     * @throws Exception 异常抛出
     */
    private static List toList(String json, ObjectMapper mapper) throws JsonProcessingException {
        if (json == null) {
            return null;
        }

        List list = mapper.readValue(json, List.class);
        for (Object obj : list) {
            if (obj instanceof String) {
                String str = (String) obj;
                if (str.startsWith("[")) {
                    obj = toList(str, mapper);
                } else if (obj.toString().startsWith("{")) {
                    obj = toMapRecursion(str, mapper);
                }
            }
        }

        return list;
    }

    /**
     * JSON字符串转换为Map，如果Map内部的value存在JSON字符串，继续解析
     *
     * @param json   JSON字符串
     * @param mapper ObjectMapper
     * @return Map对象
     * @throws Exception 异常抛出
     */
    private static Map<String, Object> toMapRecursion(String json, ObjectMapper mapper) throws JsonProcessingException {
        if (json == null) {
            return null;
        }

        Map<String, Object> map = mapper.readValue(json, Map.class);

        for (Map.Entry<String, Object> entry : map.entrySet()) {
            Object obj = entry.getValue();
            if (obj instanceof String) {
                String str = ((String) obj);
                if (str.startsWith("[")) {
                    List<?> list = toList(str, mapper);
                    map.put(entry.getKey(), list);
                } else if (str.startsWith("{")) {
                    Map<String, Object> mapRecursion = toMapRecursion(str, mapper);
                    map.put(entry.getKey(), mapRecursion);
                }
            }
        }

        return map;
    }

    /**
     * JSON字符串转换为指定类型的List
     *
     * @param jsonArray JSON字符串
     * @param clazz     目标对象类型
     * @param <T>       泛型
     * @return List对象
     * @throws Exception 异常抛出
     */
    public static <T> List<T> toList(String jsonArray, Class<T> clazz) throws JsonProcessingException {
        JavaType javaType = getCollectionType(ArrayList.class, clazz);
        return objectMapper.readValue(jsonArray, javaType);
    }


    /**
     * 获取泛型的Collection Type
     *
     * @param collectionClass 泛型的Collection
     * @param elementClasses  元素类
     * @return JavaType Java类型
     */
    public static JavaType getCollectionType(Class<?> collectionClass, Class<?>... elementClasses) {
        return objectMapper.getTypeFactory().constructParametricType(collectionClass, elementClasses);
    }

    /**
     * Map转换为JavaBean
     *
     * @param map   Map对象
     * @param clazz 目标JavaBean类型
     * @param <T>   泛型
     * @return JavaBean对象
     */
    public static <T> T mapToBean(Map map, Class<T> clazz) {
        return objectMapper.convertValue(map, clazz);
    }

    /**
     * Map转换为JSON字符串
     *
     * @param map Map对象
     * @return JSON字符串
     */
    public static String mapToJSON(Map map) throws JsonProcessingException {
        return objectMapper.writeValueAsString(map);
    }

    /**
     * Bean对象转换
     *
     * @param src   源对象
     * @param clazz 目标对象类型
     * @param <T>   泛型
     * @return 异常抛出
     */
    public static <T> T objectToAnother(Object src, Class<T> clazz) {
        return objectMapper.convertValue(src, clazz);
    }
}
