/**
 * Copyright (c) 2019, Hangzhou Mingbo Technology Co., Ltd. (sinlmao@mingbo.tech).
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.sinlmao.commons.network.soap;

import cn.sinlmao.commons.network.http.ImCharset;
import cn.sinlmao.commons.network.http.ImRequest;
import cn.sinlmao.commons.network.utils.DocumentUtil;
import com.alibaba.fastjson.JSONObject;
import org.w3c.dom.Document;

import java.util.HashMap;
import java.util.Map;

/**
 * <p> description:
 * <p> program: Sinlmao Commons Network Utils
 * <p> create: 2020-06-21 13:37
 *
 * @author Sinlmao
 * @since 1.5.1
 */
public class ImHttpSoapRequest {

    private String url;

    private String prefix = "";

    private String elementPrefix = "";
    private String elementNamespace = "";

    private ImSoapVersion version = ImSoapVersion.v_1_2;
    private ImCharset charset = ImCharset.UTF_8;

    private final Map<String, ImHttpSoapElement> bodyElements = new HashMap<>();
    private final Map<String, ImHttpSoapElement> headerElements = new HashMap<>();

    private final ImRequest request;

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * 构造基本SOAP请求
     *
     * @param url URL
     */
    public ImHttpSoapRequest(String url) {
        this.url = url;
        this.request = new ImRequest(url);
    }

    /**
     * 构造常规SOAP请求
     *
     * @param url              URL
     * @param elementPrefix    Element限定名称前缀会话级配置
     * @param elementNamespace Element限定名称命名空间会话级配置
     * @param bodyElement      SOAP Body节点元素
     */
    public ImHttpSoapRequest(String url, String elementPrefix, String elementNamespace, ImHttpSoapElement bodyElement) {
        this.url = url;
        this.elementPrefix = elementPrefix;
        this.elementNamespace = elementNamespace;
        this.bodyElements.put(bodyElement.getName().getValue(), bodyElement);
        this.request = new ImRequest(url);
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    public ImHttpSoapRequest addBodyElement(ImHttpSoapElement element) {
        appendElement(element, bodyElements);
        return this;
    }

    public Map<String, ImHttpSoapElement> getBodyElements() {
        return bodyElements;
    }

    public ImHttpSoapElement getBodyElement(String name) {
        if (!bodyElements.containsKey(name)) {
            return null;
        }
        return bodyElements.get(name);
    }

    ////////////////////////////////////////////////////////////

    public ImHttpSoapRequest addHeaderElement(ImHttpSoapElement element) {
        appendElement(element, headerElements);
        return this;
    }

    public Map<String, ImHttpSoapElement> getHeaderElements() {
        return headerElements;
    }

    public ImHttpSoapElement getHeaderElement(String name) {
        if (!headerElements.containsKey(name)) {
            return null;
        }
        return headerElements.get(name);
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    public ImHttpSoapRequest addRequestHeader(String name, String value) {
        request.addHeader(name, value);
        return this;
    }

    public ImHttpSoapRequest addRequestHeader(Map<String, String> headers) {
        request.addHeader(headers);
        return this;
    }

    /**
     * @param headers
     * @return
     * @deprecated v1.6.0
     */
    @Deprecated
    public ImHttpSoapRequest addRequestHeader(JSONObject headers) {
        request.addHeader(headers);
        return this;
    }

    public ImHttpSoapRequest setRequestHeader(String name, String value) {
        request.setHeader(name, value);
        return this;
    }

    public ImHttpSoapRequest setRequestHeader(Map<String, String> headers) {
        request.setHeader(headers);
        return this;
    }

    /**
     * @param headers
     * @return
     * @deprecated v1.6.0
     */
    @Deprecated
    public ImHttpSoapRequest setRequestHeader(JSONObject headers) {
        request.setHeader(headers);
        return this;
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    public ImHttpSoapRequest enableProxyServer(int port) {
        request.enableProxyServer(port);
        return this;
    }

    public ImHttpSoapRequest enableProxyServer(String host, int port) {
        request.enableProxyServer(host, port);
        return this;
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    public String getUrl() {
        return url;
    }

    public ImHttpSoapRequest setUrl(String url) {
        this.url = url;
        this.request.setUrl(url);
        return this;
    }

    public String getPrefix() {
        return prefix;
    }

    public ImHttpSoapRequest setPrefix(String prefix) {
        this.prefix = prefix;
        return this;
    }

    public String getElementPrefix() {
        return elementPrefix;
    }

    public ImHttpSoapRequest setElementPrefix(String elementPrefix) {
        this.elementPrefix = elementPrefix;
        return this;
    }

    public ImHttpSoapRequest setElementNamespace(String elementNamespace) {
        this.elementNamespace = elementNamespace;
        return this;
    }

    public ImHttpSoapRequest setElementNamespace(String elementPrefix, String elementNamespace) {
        this.elementPrefix = elementPrefix;
        this.elementNamespace = elementNamespace;
        return this;
    }

    public String getElementNamespace() {
        return elementNamespace;
    }

    public ImSoapVersion getVersion() {
        return version;
    }

    public ImHttpSoapRequest setVersion(ImSoapVersion version) {
        this.version = version;
        return this;
    }

    public ImCharset getCharset() {
        return charset;
    }

    /**
     * 设置Charset编码
     * <p>
     * 默认为UTF-8
     * <p>
     * <font color="#666666">Set the Charset</font>
     * <p>
     * <font color="#666666">Default is UTF-8</font>
     *
     * @param charset 编码 <br/> <font color="#666666">Charset</font>
     * @return ImRequest对象实体 <br/> <font color="#666666">ImRequest object entity</font>
     */
    public ImHttpSoapRequest setCharset(ImCharset charset) {
        this.charset = charset;
        this.request.setCharset(charset);
        return this;
    }

    /**
     * 获取ImRequest（HTTP）对象实体
     * <p>
     * <font color="#666666">Get the ImRequest (HTTP) object entity</font>
     * <p>
     *
     * @return ImRequest对象实体 <br/> <font color="#666666">ImRequest object entity</font>
     */
    public ImRequest getRequest() {
        return request;
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    private void appendElement(ImHttpSoapElement element, Map<String, ImHttpSoapElement> elements) {
        //假如数据有效
        if (element != null) {
            //是否允许继承
            if (element.getName().isInherit()) {
                //如果不包含前缀设置，则继承全局设置（SOAP消息规范）
                if (element.getName().getPrefix().isEmpty()) {
                    element.getName().setPrefix(elementPrefix);
                }
                if (element.getName().getNamespace().isEmpty()) {
                    element.getName().setNamespace(elementNamespace);
                }
            }
            //放入元素集合
            elements.put(element.getName().getValue(), element);
        }
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * 获取消息协议的Document类型数据
     *
     * @return 消息协议的Document类型数据
     * @since 1.5.3
     */
    public Document getMessageDataToDocument() {
        return ImHttpSoapClient.resolve(this);
    }

    /**
     * 获取消息协议的String类型数据
     *
     * @return 消息的String类型数据
     * @since 1.5.3
     */
    public String getMessageDataToString() {
        return DocumentUtil.toString(ImHttpSoapClient.resolve(this));
    }
}
