/**
 * Copyright (c) 2019, Hangzhou Mingbo Technology Co., Ltd. (sinlmao@mingbo.tech).
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.sinlmao.commons.network.soap;

import cn.sinlmao.commons.network.http.*;
import cn.sinlmao.commons.network.utils.DocumentUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * <b>SOAP协议消息客户实现类</b>
 * <p>
 * 用于发送SOAP消息协议请求，所有请求的数据都需要封装在ImHttpSoapRequest中，此外还负责组装、解析SOAP消息数据
 * <br/><br/>
 * <b>SOAP protocol message client implementation class</b>
 * <p>
 * Used to send SOAP message protocol requests. All requested data needs to be encapsulated in ImHttpSoapRequest.
 * In addition, it is also responsible for assembling and parsing SOAP message data.
 * <br/><br/>
 * <p> description: SOAP协议消息客户实现类
 * <p> program: Sinlmao Commons Network Utils
 * <p> create: 2020-06-21 13:25
 *
 * @author Sinlmao
 * @since 1.5.1
 */
public final class ImHttpSoapClient {

    /**
     * 发起SOAP协议消息请求
     * <p>
     * <font color="#666666">Initiate SOAP protocol message request</font>
     *
     * @param request ImHttpSoapRequest SOAP协议消息请求 <br/> <font color="#666666">ImHttpSoapRequest SOAP protocol message request</font>
     * @return ImResponse 请求应答 <br/> <font color="#666666">Request reply</font>
     * @throws Exception 异常类 <br/> <font color="#666666">Exception class</font>
     * @see ImHttpSoapRequest
     * @see ImResponse
     */
    public static ImResponse soap(ImHttpSoapRequest request) throws Exception {

        //解析SOAP消息数据
        Document document = resolve(request);

        //获得ImHttpClient的ImRequest对象
        ImRequest imRequest = request.getRequest();
        //设置方法
        imRequest.setMethod(ImMethod.POST);
        //输入SOAP格式的数据
        imRequest.setInputData(DocumentUtil.toString(document, imRequest.getCharset(), false));

        //根据SOAP版本设置内容类型
        if (request.getVersion() == ImSoapVersion.v_1_1) {
            imRequest.setContentType(ImContentType.SOAP_1_1);
        }
        if (request.getVersion() == ImSoapVersion.v_1_2) {
            imRequest.setContentType(ImContentType.SOAP_1_2);
        }

        //发起并返回SOAP协议的请求和结果
        return ImHttpClient.send(imRequest);
    }

    /**
     * 解析并组装SOAP协议消息数据
     * <p>
     * <font color="#666666">Parse and assemble SOAP protocol message data</font>
     *
     * @param request ImHttpSoapRequest SOAP协议消息请求 <br/> <font color="#666666">ImHttpSoapRequest SOAP protocol message request</font>
     * @return XML消息数据 <br/> <font color="#666666">XML message data</font>
     * @see ImHttpSoapRequest
     * @see Document
     */
    public static Document resolve(ImHttpSoapRequest request) {

        //创建文档
        Document document = DocumentUtil.buildeDocument();

        //创建Envelope节点
        Element envelopeElement = document.createElement(getTagName("Envelope", request.getPrefix(),
                request.getVersion(), true));
        //设置SOAP协议的一些基本属性
        envelopeElement.setAttribute("xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance");
        envelopeElement.setAttribute("xmlns:xsd", "http://www.w3.org/2001/XMLSchema");
        //如果没有设置前缀，根据版本匹配
        if (request.getPrefix().isEmpty()) {
            //SOAP1.1和1.2协议不同，相关属性区分设置
            if (request.getVersion() == ImSoapVersion.v_1_1) {
                envelopeElement.setAttribute("xmlns:SOAP-ENV", "http://schemas.xmlsoap.org/soap/envelope/");
            }
            if (request.getVersion() == ImSoapVersion.v_1_2) {
                envelopeElement.setAttribute("xmlns:env", "http://www.w3.org/2003/05/soap-envelope");
            }
        } else {
            //SOAP1.1和1.2协议不同，相关属性区分设置
            if (request.getVersion() == ImSoapVersion.v_1_1) {
                envelopeElement.setAttribute("xmlns:" + request.getPrefix(), "http://schemas.xmlsoap.org/soap/envelope/");
            }
            if (request.getVersion() == ImSoapVersion.v_1_2) {
                envelopeElement.setAttribute("xmlns:" + request.getPrefix(), "http://www.w3.org/2003/05/soap-envelope");
            }
        }
        //如果额外设置了元素命名空间和前缀
        if (!request.getElementPrefix().isEmpty() && !request.getElementNamespace().isEmpty()) {
            envelopeElement.setAttribute(getXmlnsName(request.getElementPrefix()), request.getElementNamespace());
        }
        //置入节点数据
        document.appendChild(envelopeElement);

        //创建Element Qname全局设置
        ImSoapQname elementQname = new ImSoapQname(envelopeElement.getTagName(), request.getElementPrefix(), request.getElementNamespace());

        //处理Header数据
        Map<String, ImHttpSoapElement> headerElements = request.getHeaderElements();
        if (headerElements.size() > 0) {
            //创建Header节点
            Element headerElement = document.createElement(getTagName("Header", request.getPrefix(),
                    request.getVersion(), true));

            //处理节点数据
            Collection<ImHttpSoapElement> childElements = headerElements.values();
            for (ImHttpSoapElement childElement : childElements) {
                //置入节点数据
                headerElement.appendChild(resolveElement(childElement, document, request, envelopeElement,
                        childElement.getName().isQualified() ? childElement.getName() : elementQname));
            }

            //置入Header节点数据
            envelopeElement.appendChild(headerElement);
        }

        //创建Body节点
        Element bodyElement = document.createElement(getTagName("Body", request.getPrefix(), request.getVersion(), true));
        //置入节点数据
        envelopeElement.appendChild(bodyElement);

        //如果存在BodyElement节点
        if (request.getBodyElements().size() > 0) {
            Collection<ImHttpSoapElement> childElements = request.getBodyElements().values();
            for (ImHttpSoapElement childElement : childElements) {
                //置入节点数据
                bodyElement.appendChild(resolveElement(childElement, document, request, envelopeElement,
                        childElement.getName().isQualified() ? childElement.getName() : elementQname));
            }
        }

        return document;
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * 解析元素节点数据
     * <p>
     * <font color="#666666">Resolve element node data</font>
     *
     * @param soapElement     待解析的节点对象（ImHttpSoapElement） <br/> <font color="#666666">Node object to be parsed (ImHttpSoapElement)</font>
     * @param document        文档对象（Document） <br/> <font color="#666666">Document object (Document)</font>
     * @param request         SOAP请求对象 <br/> <font color="#666666">SOAP request object</font>
     * @param envelopeElement 根节点数据（envelope节点） <br/> <font color="#666666">Root node data (envelope node)</font>
     * @param qname           限定名称对象 <br/> <font color="#666666">Qualified name object</font>
     * @return 元素节点数据 <br/> <font color="#666666">Element node data</font>
     * @see ImHttpSoapElement
     * @see Document
     * @see ImHttpSoapRequest
     * @see ImSoapQname
     */
    public static Element resolveElement(ImHttpSoapElement soapElement, Document document, ImHttpSoapRequest request,
                                          Element envelopeElement, ImSoapQname qname) {

        //获得Prefix和Namespace
        String prefix = soapElement.getName().getPrefix().isEmpty() ? (soapElement.getName().isInherit() ? qname.getPrefix() : "") : soapElement.getName().getPrefix();
        String namespace = soapElement.getName().getNamespace().isEmpty() ? (soapElement.getName().isInherit() ? qname.getNamespace() : "") : soapElement.getName().getNamespace();

        //创建Element
        Element element = document.createElement(getTagName(soapElement.getName().getValue(), prefix, request.getVersion()));

        //处理命名空间
        element.setAttribute(getXmlnsName(prefix), namespace);

        //当前节点指定前缀且指定命名空间，且命名空间的值和全局设置的Element Namespace不相同时，需要在根节点（envelope节点）额外声明命名空间
        if (!soapElement.getName().getPrefix().isEmpty()
                && soapElement.getName().getNamespace().equals(request.getElementNamespace())) {
            //在envelope节点（根节点）加以声明
            envelopeElement.setAttribute(getXmlnsName(soapElement.getName().getPrefix()), soapElement.getName().getNamespace());
        }

        //开始处理节点数据
        //如果节点数据指定的是一个子节点
        if (soapElement.getValue() instanceof ImHttpSoapElement) {
            //置入子节点，子节点数据递归调用本方法（resolveElement）解析
            element.appendChild(resolveElement((ImHttpSoapElement) soapElement.getValue(), document, request, envelopeElement,
                    soapElement.getName().isQualified() ? soapElement.getName() : qname));
        }
        //如果节点数据指定的是一个子节点数据集（List<ImHttpSoapElement>）
        else if (soapElement.getValue() instanceof List) {
            //提取子节点数据集合
            @SuppressWarnings("unchecked")
            List<ImHttpSoapElement> childValues = (List<ImHttpSoapElement>) soapElement.getValue();
            //循环并递归调用本方法（resolveElement）解析
            for (ImHttpSoapElement childValue : childValues) {
                //置入子节点
                element.appendChild(resolveElement(childValue, document, request, envelopeElement,
                        childValue.getName().isQualified() ? childValue.getName() : qname));
            }
        }
        //如果节点数据是值数据，则直接toString
        else {
            if (soapElement.getValue() != null) {
                element.setTextContent(soapElement.getValue().toString());
            }
        }

        return element;
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /**
     * 获取SOAP xmlns
     * <p>
     * <font color="#666666">Get SOAP xmlns</font>
     *
     * @return xmlns值 <br/> <font color="#666666">xmlns value</font>
     */
    private static String getXmlnsName() {
        return getXmlnsName("");
    }

    /**
     * 获取SOAP xmlns
     * <p>
     * <font color="#666666">Get SOAP xmlns</font>
     *
     * @param prefix 限定名称的前缀 <br/> <font color="#666666">Qualified name prefix</font>
     * @return xmlns值 <br/> <font color="#666666">xmlns value</font>
     */
    private static String getXmlnsName(String prefix) {
        if (prefix.isEmpty()) {
            return "xmlns";
        } else {
            return "xmlns:" + prefix;
        }
    }

    /**
     * 获取SOAP xml节点名称
     * <p>
     * <font color="#666666">Get SOAP xml node name</font>
     *
     * @param name    名称 <br/> <font color="#666666">Name</font>
     * @param prefix  限定名称的前缀 <br/> <font color="#666666">Qualified name prefix</font>
     * @param version SOAP版本 <br/> <font color="#666666">SOAP version</font>
     * @return xml节点名称 <br/> <font color="#666666">xml node name</font>
     */
    private static String getTagName(String name, String prefix, ImSoapVersion version) {
        return getTagName(name, prefix, version, false);
    }

    /**
     * 获取SOAP xml节点名称
     * <p>
     * <font color="#666666">Get SOAP xml node name</font>
     *
     * @param name      名称 <br/> <font color="#666666">Name</font>
     * @param prefix    限定名称的前缀 <br/> <font color="#666666">Qualified name prefix</font>
     * @param version   SOAP版本 <br/> <font color="#666666">SOAP version</font>
     * @param isDefault 是否默认 <br/> <font color="#666666">Whether default</font>
     * @return xml节点名称 <br/> <font color="#666666">xml node name</font>
     */
    private static String getTagName(String name, String prefix, ImSoapVersion version, boolean isDefault) {
        if (prefix != null && !prefix.isEmpty()) {
            return prefix + ":" + name;
        } else {
            if (isDefault) {
                String default_prefix = "env:";
                if (version == ImSoapVersion.v_1_1) {
                    default_prefix = "SOAP-ENV:";
                }
                return default_prefix + name;
            } else {
                return name;
            }
        }
    }
}
