/**
 * Copyright (c) 2019, Sinlmao (888@1st.com).
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.sinlmao.commons.network.utils;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.*;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.*;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Properties;

/**
 * <p> program: Sinlmao Commons Network Utils
 * <p> description: Document工具类
 * <p> create: 2020-06-22 16:46
 *
 * @author Sinlmao
 * @since 1.5.1
 */
public class DocumentUtil {


    /**
     * 构建Document对象
     *
     * @return Document对象
     */
    public static Document buildeDocument() {
        try {
            return newBuilder().newDocument();
        } catch (ParserConfigurationException e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * 构建Document带root节点对象
     *
     * @param node Node阶段对象
     * @return Document对象
     */
    public static Document newXMLDocument(Node node) {
        Document doc = buildeDocument();
        doc.appendChild(doc.importNode(node, true));
        return doc;
    }

    /**
     * 创建DocumentBuilder对象
     *
     * @return DocumentBuilder对象
     * @throws ParserConfigurationException 异常
     */
    public static DocumentBuilder newBuilder()
            throws ParserConfigurationException {
        return newBuilderFactory().newDocumentBuilder();
    }

    /**
     * 创建DocumentBuilderFactory
     *
     * @return DocumentBuilderFactory对象
     */
    public static DocumentBuilderFactory newBuilderFactory() {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(true);
        return factory;
    }

    /**
     * XML字符串转换为Document对象
     *
     * @param xml 遵循XML规范的字符串
     * @return Document对象
     */
    public static Document parseDocument(String xml) {
        if (xml == null) {
            throw new IllegalArgumentException();
        }
        try {
            return newBuilder().parse(new InputSource(new StringReader(xml)));
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * InputStream转换为Document对象
     *
     * @param input 输入流
     * @return Document对象
     */
    public static Document parseDocument(InputStream input) {
        if (input == null) {
            throw new IllegalArgumentException();
        }
        try {
            return newBuilder().parse(input);
        } catch (Exception e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * 从文件全名（含路径）的指向的文件数据转换为Document对象
     *
     * @param fileName 文件全名（含路径）
     * @return Document对象
     */
    public static Document loadDocumentFromFile(String fileName) {
        if (fileName == null) {
            throw new IllegalArgumentException("未指定文件名及其物理路径！");
        }
        try {
            return newBuilder().parse(new File(fileName));
        } catch (SAXException e) {
            throw new IllegalArgumentException("目标文件（" + fileName + "）不能被正确解析为XML！" + e.getMessage());
        } catch (IOException e) {
            throw new IllegalArgumentException("不能获取目标文件（" + fileName + "）！" + e.getMessage());
        } catch (ParserConfigurationException e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * Document对象转成字符串
     *
     * @param document Document对象
     * @return 字符串
     */
    public static String toString(Document document) {
        return toString(document, StandardCharsets.UTF_8.name(), true);
    }

    /**
     * Document对象转成字符串
     *
     * @param document Document对象
     * @param charset  字符编码
     * @return 字符串
     */
    public static String toString(Document document, String charset) {
        return toString(document, charset, true);
    }

    /**
     * Document对象转成字符串
     *
     * @param document Document对象
     * @param charset  字符编码
     * @param indent   是否格式化（缩进、换行等）
     * @return 字符串
     */
    public static String toString(Document document, String charset, boolean indent) {
        String result = null;
        if (document != null) {
            StringWriter strWtr = new StringWriter();
            StreamResult strResult = new StreamResult(strWtr);
            try {
                Transformer transformer = newTransformer(charset, indent);
                transformer.transform(new DOMSource(document.getDocumentElement()), strResult);
            } catch (Exception e) {
                System.err.println("XML.toString(Document): " + e);
            }
            result = strResult.getWriter().toString();
            try {
                strWtr.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return result;
    }

    /**
     * Node对象转成字符串
     *
     * @param node Node对象
     * @return 字符串
     */
    public static String toNodeString(Node node) {
        return toNodeString(node, StandardCharsets.UTF_8.name(), true);
    }

    /**
     * Node对象转成字符串
     *
     * @param node    Node对象
     * @param charset 字符编码
     * @return 字符串
     */
    public static String toNodeString(Node node, String charset) {
        return toNodeString(node, charset, true);
    }

    /**
     * Node对象转成字符串
     *
     * @param node    Node对象
     * @param charset 字符编码
     * @param indent  是否格式化（缩进、换行等）
     * @return 字符串，如果失败则返回一个空字符串""
     */
    public static String toNodeString(Node node, String charset, boolean indent) {
        if (node == null) {
            throw new IllegalArgumentException();
        }
        Transformer transformer = newTransformer(charset, indent);
        try {
            StringWriter sw = new StringWriter();
            transformer.transform(new DOMSource(node), new StreamResult(sw));
            return sw.toString();
        } catch (TransformerException e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * 创建Transformer对象
     *
     * @return Transformer
     */
    public static Transformer newTransformer(String charset, boolean indent) {
        try {
            Transformer transformer = TransformerFactory.newInstance().newTransformer();
            Properties properties = transformer.getOutputProperties();
            properties.setProperty(OutputKeys.ENCODING, charset);
            properties.setProperty(OutputKeys.METHOD, "xml");
            properties.setProperty(OutputKeys.VERSION, "1.0");
            if (indent) {
                properties.setProperty(OutputKeys.INDENT, "yes");
                properties.setProperty("{http://xml.apache.org/xslt}indent-amount", "4");
            }
            transformer.setOutputProperties(properties);
            return transformer;
        } catch (TransformerConfigurationException e) {
            throw new RuntimeException(e.getMessage());
        }
    }
}
