/**
 * Copyright (c) 2019, Sinlmao (888@1st.com).
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.sinlmao.commons.network.ssl;

import javax.net.ssl.KeyManager;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;

/**
 * program Sinlmao Commons Network Utils
 * description
 * create 2020-06-18 11:20
 *
 * @author Sinlmao
 */
public class TrustAllCertificateContextBuilder {

    /**
     * 以指定协议创建一个信任所有证书的SSLSocketFactory
     *
     * @param protocol 指定的协议
     * @return SSLSocketFactory
     * @throws NoSuchAlgorithmException NoSuchAlgorithmException
     * @throws KeyManagementException   KeyManagementException
     */
    public static SSLSocketFactory build(String protocol) throws Exception {
        return build(protocol, null, new SecureRandom());
    }

    /**
     * 以默认协议创建一个带客户端证书的SSLSocketFactory
     *
     * @param keyManagers 客户端证书管理器
     * @return SSLSocketFactory
     * @throws NoSuchAlgorithmException NoSuchAlgorithmException
     * @throws KeyManagementException   KeyManagementException
     */
    public static SSLSocketFactory build(KeyManager[] keyManagers) throws Exception {
        return build(ProtocolSSLFactory.PROTOCOL_DEFAULT, keyManagers, null);
    }

    /**
     * 以指定协议且携带客户端证书创建一个带客户端证书的SSLSocketFactory
     *
     * @param protocol    指定的协议
     * @param keyManagers 客户端证书管理器
     * @return SSLSocketFactory
     * @throws NoSuchAlgorithmException NoSuchAlgorithmException
     * @throws KeyManagementException   KeyManagementException
     */
    public static SSLSocketFactory build(String protocol, KeyManager[] keyManagers) throws Exception {
        return build(protocol, keyManagers, null);
    }

    /**
     * 【完整实现】以指定协议、指定SecureRandom，并且携带客户端证书创建一个带客户端证书的SSLSocketFactory
     *
     * @param protocol     指定的协议
     * @param keyManagers  客户端证书管理器
     * @param secureRandom 指定的SecureRandom
     * @return SSLSocketFactory
     * @throws NoSuchAlgorithmException NoSuchAlgorithmException
     * @throws KeyManagementException   KeyManagementException
     */
    public static SSLSocketFactory build(String protocol, KeyManager[] keyManagers, SecureRandom secureRandom) throws Exception {
        TrustManager[] trustAllCerts = new TrustManager[1];
        trustAllCerts[0] = new TrustAllCertificateManager();
        return ProtocolSSLFactory.getInstance(keyManagers, trustAllCerts, secureRandom);
    }
}
